package com.xebialabs.deployit.plugin.jbossas.step;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.generic.freemarker.ArtifactUploader;
import com.xebialabs.deployit.plugin.generic.freemarker.CiAwareObjectWrapper;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.overthere.DefaultProcessOutputHandler;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.*;
import com.xebialabs.overthere.util.MultipleOverthereProcessOutputHandler;
import com.xebialabs.overthere.util.OverthereUtils;
import com.xebialabs.xlplatform.satellite.Satellite;
import com.xebialabs.xlplatform.satellite.SatelliteAware;
import freemarker.template.Configuration;
import freemarker.template.TemplateException;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Collections;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Maps.newHashMap;

public abstract class TwiddleStep implements SatelliteAware{

    private transient OverthereConnection remoteConn;
    private transient OverthereFile remoteWorkingDir;

    private final HostContainer container;
    private final String scriptTemplatePath;
    protected transient ExecutionContext ctx;

    public TwiddleStep(HostContainer container, String scriptTemplatePath) {
        this.container = container;
        this.scriptTemplatePath = checkNotNull(scriptTemplatePath, "scriptTemplatePath is null");
    }

    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        try {
            this.ctx = ctx;
            String osSpecificTemplate = resolveOsSpecificTemplate();
            String executableContent = evaluateTemplate(osSpecificTemplate, getVars());
            logger().debug(executableContent);
            OverthereFile executable = uploadExecutable(executableContent, OverthereUtils.getName(osSpecificTemplate));
            CmdLine cmdLine = CmdLine.build(executable.getPath());
            TwiddleProcessOutputHandler parser = new TwiddleProcessOutputHandler();
            MultipleOverthereProcessOutputHandler handle = MultipleOverthereProcessOutputHandler.multiHandler(new DefaultProcessOutputHandler(ctx), parser);
            ctx.logOutput("Executing " + executable.getPath() + " on host " + container.getHost());
            
            int rc = getRemoteConnection().execute(handle, cmdLine);
            if (rc == 0) {
                doHandle(parser);
                return StepExitCode.SUCCESS;
            }
            return StepExitCode.FAIL;
        } finally {
            disconnect();
        }
    }

    protected void disconnect() {
        if (remoteConn != null) {
            remoteConn.close();
        }

        remoteWorkingDir = null;
        remoteConn = null;
        ctx = null;
    }
    protected Map<String, Object> getVars() {
        return Collections.emptyMap();
    }

    protected void doHandle(@SuppressWarnings("unused") TwiddleProcessOutputHandler handler) {
    }

    private String resolveOsSpecificTemplate() {
        String osSpecificScript = scriptTemplatePath;
        if (scriptTemplatePath.lastIndexOf('.') == -1) {
            OperatingSystemFamily os = container.getHost().getOs();
            osSpecificScript = osSpecificScript + os.getScriptExtension();
        }

        if (!classpathResourceExists(osSpecificScript)) {
            throw new IllegalArgumentException("Resource " + osSpecificScript + " not found in classpath");
        }

        return osSpecificScript;
    }

    private String evaluateTemplate(String templatePath, Map<String, Object> vars) {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        try {
            freemarker.template.Template template = cfg.getTemplate(templatePath);
            StringWriter sw = new StringWriter();
            template.createProcessingEnvironment(vars, sw, new CiAwareObjectWrapper(new WorkingFolderUploader(), false)).process();
            return sw.toString();
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } catch (TemplateException e) {
            logger().error("evaluate Template error", e);
            throw new RuntimeException(e);
        }
    }

    private OverthereConnection getRemoteConnection() {
        if (remoteConn == null) {
            remoteConn = container.getHost().getConnection();
        }
        return remoteConn;
    }

    private OverthereFile getRemoteWorkingDirectory() {
        if (remoteWorkingDir == null) {
            OverthereFile tempDir = getRemoteConnection().getTempFile("jboss_plugin", ".tmp");
            tempDir.mkdir();
            remoteWorkingDir = tempDir;
        }
        return remoteWorkingDir;
    }

    private OverthereFile uploadExecutable(String content, String fileName) {
        OverthereFile targetExecutable = getRemoteWorkingDirectory().getFile(fileName);
        OverthereUtils.write(content.getBytes(), targetExecutable);
        targetExecutable.setExecutable(true);
        return targetExecutable;
    }

    private static boolean classpathResourceExists(String resource) {
        return Thread.currentThread().getContextClassLoader().getResource(resource) != null;
    }

    @Override
    public Satellite getSatellite() {
        return container.getHost().getSatellite();
    }

    private class WorkingFolderUploader implements ArtifactUploader {
        private Map<String, String> uploadedFiles = newHashMap();

        @Override
        public String upload(Artifact file) {
            if (uploadedFiles.containsKey(file.getName())) {
                return uploadedFiles.get(file.getName());
            }
            uploadedFiles.put(file.getName(), file.getFile().getPath());
            return file.getFile().getPath();

        }
    }

    public HostContainer getContainer() {
        return container;
    }

    protected abstract Logger logger();


}
