package com.xebialabs.deployit.plugin.jbossdm.container;

import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.inspection.InspectionProperty;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.jbossdm.controltask.ControlTaskDispatcher;
import com.xebialabs.deployit.plugin.jbossdm.step.CliDaemon;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.RuntimeIOException;

import java.util.List;

import static com.google.common.collect.Lists.newArrayList;

@Metadata(description = "JBoss Cli Managed Container")
public abstract class CliBasedContainer extends BaseContainer implements ApplicationContainer, ResourceContainer, CliManagedContainer, CliManagingContainer {

    @Property(asContainment = true)
    @InspectionProperty
    private Host host;

    @Property(description = "JBoss home directory")
    @InspectionProperty
    private String home;

    @Property(required = false, defaultValue = "9999", label = "Administrative port", description = "TCP port which is used to login to JBoss Native Administration, default is 9999")
    @InspectionProperty(required = false)
    private int port;

    @Property(required = false, defaultValue="true", description = "Connection to CLI is setup using a daemon. Set to false if host connection does not support streaming.")
    @InspectionProperty(required = false)
    private boolean enableDaemon;

    @Property(required = false, defaultValue="localhost", label = "Administrative Host", description = "Host which is used to login to JBoss Native Administration, default is localhost")
    @InspectionProperty(required = false)
    private String adminHostAddress;

    @Property(required = false, label = "Administrative username", description = "Username which is used to login to JBoss Native Administration.")
    @InspectionProperty(required = false)
    private String username;

    @Property(required = false, password = true, label = "Administrative password", description = "Password which is used to login to JBoss Native Administration.")
    @InspectionProperty(required = false)
    private String password;

    @Property(defaultValue = "jboss-cli", description = "JBoss CLI script prefix. Depending on the host, either an '.sh' or '.bat' will be appended to get the cli script name.")
    @InspectionProperty(required = false)
    private String cliScriptPrefix;

    @Property(hidden = true, required = false, description = "List of python library scripts that should be automatically loaded when using a JBoss CLI script.")
    private List<String> libraries = newArrayList();

    public Host getHost() {
        return host;
    }

    @Override
    public String getCliExecutable() {
        OperatingSystemFamily os = host.getOs();
        return getHome() + os.getFileSeparator() +"bin" + os.getFileSeparator() +  getCliScriptPrefix() + os.getScriptExtension();
    }

    public void setHost(Host host) {
        this.host = host;
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    @Override
    public String getAdminHostAddress() {
        return adminHostAddress;
    }

    public void setAdminHostAddress(String adminHostAddress) {
        this.adminHostAddress = adminHostAddress;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public String getCliScriptPrefix() {
        return cliScriptPrefix;
    }

    public void setCliScriptPrefix(String cliScriptPrefix) {
        this.cliScriptPrefix = cliScriptPrefix;
    }

    public String getHome() {
        return home;
    }

    public void setHome(String home) {
        this.home = home;
    }

    public List<String> getLibraries() {
        return libraries;
    }

    public void setLibraries(List<String> libraries) {
        this.libraries = libraries;
    }

    @Override
    public CliManagingContainer getManagingContainer() {
        return this;
    }

	@Override
	public Object execute(ExecutionContext ctx, String cmd) {
		String attrName = "managingContainersDaemon_" + this.getName();
		CliDaemon daemon = (CliDaemon)ctx.getAttribute(attrName);
		if (daemon == null) {
			daemon = new CliDaemon(getCliExecutable(), getUsername(), getPassword(), getAdminHostAddress(), getPort(), getHost(), ctx);
			ctx.setAttribute(attrName, daemon);
		}

		try {
            if (isEnableDaemon()) {
			    return daemon.executeCliCommand(cmd);
            } else {
                return daemon.executeCliCommandWithDaemon(cmd);
            }
		} catch(RuntimeIOException e) {
			daemon.disconnect();
			ctx.setAttribute(attrName, null);
			throw e;
		}
	}

    public List<Step> controlTaskDispatch(String name) {
        return ControlTaskDispatcher.dispatch(name, this);
    }


    @Override
    public String getProfile() {
        return "default";
    }

    public boolean isEnableDaemon() {
        return enableDaemon;
    }

    public void setEnableDaemon(boolean enableDaemon) {
        this.enableDaemon = enableDaemon;
    }
}
