package com.xebialabs.deployit.plugin.jbossdm.deployed;

import com.google.common.base.Strings;
import com.xebialabs.deployit.plugin.api.deployment.planning.*;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionPlanningContext;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployed;
import com.xebialabs.deployit.plugin.jbossdm.container.CliManagedContainer;
import com.xebialabs.deployit.plugin.jbossdm.controltask.ControlTaskDispatcher;
import com.xebialabs.deployit.plugin.jbossdm.step.CliDeploymentStep;
import com.xebialabs.deployit.plugin.jbossdm.step.CliInspectionStep;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

@Metadata(virtual = true, description = "Base for all deployed that utilize the JBoss Cli for configuration")
public class CliManagedDeployed<D extends Deployable, C extends CliManagedContainer> extends BaseDeployed<D,C> {

    @Property(hidden = true, defaultValue = "50", description = "The order of the step in the step list for the create operation.")
    private int createOrder;

    @Property(hidden = true, defaultValue = "40", description = "The order of the step in the step list for the destroy operation.")
    private int destroyOrder;

    @Property(hidden = true, defaultValue = "50", description = "The order of the step in the step list for the modify operation.")
    private int modifyOrder;

    @Property(hidden = true, defaultValue = "50", description = "The order of the step in the step list for the noop operation.")
    private int noopOrder;

    @Property(hidden = true, defaultValue = "Create")
    private String createVerb;

    @Property(hidden = true, defaultValue = "Modify")
    private String modifyVerb;

    @Property(hidden = true, defaultValue = "Destroy")
    private String destroyVerb;

    @Property(hidden = true, defaultValue = "Modify")
    private String noopVerb;

    @Property(required = true, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the create operation.")
    private String createScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the modify operation.")
    private String modifyScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the destroy operation.")
    private String destroyScript;

    @Property(required = false, hidden = true, description = "Classpath to the script that is uploaded and executed on the generic container for the noop operation.")
    private String noopScript;

    @Property(required = false, hidden = true, description = "Classpath to the script used to inspect the generic container.")
    private String inspectScript;

    @Property(hidden = true, required = false, description = "List of python library scripts that should be automatically loaded when using a JBoss CLI script.")
    private List<String> libraries = newArrayList();

    protected boolean addStep(DeploymentPlanningContext ctx, String script, int order, String verb, Delta delta) {
        if (!Strings.nullToEmpty(script).trim().isEmpty()) {
            HashMap<String, Object> pyCtx = createPythonVars();
            pyCtx.put("delta", delta);
            pyCtx.put("deployedApplication",ctx.getDeployedApplication());
            CliDeploymentStep step = new CliDeploymentStep(script, order, pyCtx, getDescription(verb), this.getContainer());
            step.setAdditionalLibraries(libraries);
            ctx.addStep(step);
            return true;
        }

        return false;
    }

    protected HashMap<String, Object> createPythonVars() {
        HashMap<String,Object> pyCtx = newHashMap();
        pyCtx.put("deployed", this);
        pyCtx.put("container", this.getContainer());
        return pyCtx;
    }

    public List<Step> controlTaskDispatch(String name) {
        return ControlTaskDispatcher.dispatch(name, this);
    }

    public String getDescription(String verb) {
        return String.format("%s %s on %s", verb, getName(), getContainer().getName());
    }

    @Inspect
    public void inspect(InspectionPlanningContext ctx) {
        if (!Strings.nullToEmpty(getInspectScript()).trim().isEmpty()) {
            CliInspectionStep step = new CliInspectionStep(getInspectScript(), 0, createPythonVars(), "Inspect " + getName(), this.getContainer());
            step.setAdditionalLibraries(libraries);
            ctx.addStep(step);
        }
    }


    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta delta) {
        CliManagedDeployed<D, C> deployed = (CliManagedDeployed<D, C>) delta.getDeployed();
        deployed.addStep(ctx, getCreateScript(), getCreateOrder(), getCreateVerb(), delta);
    }

    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx, Delta delta) {
        CliManagedDeployed<D, C> deployed = (CliManagedDeployed<D, C>) delta.getPrevious();
        deployed.addStep(ctx, getDestroyScript(), getDestroyOrder(), getDestroyVerb(), delta);
    }

    @Noop
    public void executeNoop(DeploymentPlanningContext ctx, Delta delta) {
        CliManagedDeployed<D, C> deployed = (CliManagedDeployed<D, C>) delta.getDeployed();
        deployed.addStep(ctx, getNoopScript(), getNoopOrder(), getNoopVerb(), delta);
    }

    @Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta delta) {
        CliManagedDeployed<D, C> deployed = (CliManagedDeployed<D, C>) delta.getDeployed();
        if (!deployed.addStep(ctx, getModifyScript(), getModifyOrder(), getModifyVerb(), delta)) {
            CliManagedDeployed<D, C> previousDeployed = (CliManagedDeployed<D, C>) delta.getPrevious();
            previousDeployed.executeDestroy(ctx, delta);
            deployed.executeCreate(ctx, delta);
        };
    }

    public List<String> getLibraries() {
        return libraries;
    }

    public void setLibraries(List<String> libraries) {
        this.libraries = libraries;
    }

    public String getCreateScript() {
        return createScript;
    }

    public void setCreateScript(String createScript) {
        this.createScript = createScript;
    }

    public String getModifyScript() {
        return modifyScript;
    }

    public void setModifyScript(String modifyScript) {
        this.modifyScript = modifyScript;
    }

    public String getDestroyScript() {
        return destroyScript;
    }

    public void setDestroyScript(String destroyScript) {
        this.destroyScript = destroyScript;
    }

    public String getNoopScript() {
        return noopScript;
    }

    public void setNoopScript(String noopScript) {
        this.noopScript = noopScript;
    }

    public int getCreateOrder() {
        return createOrder;
    }

    public void setCreateOrder(int createOrder) {
        this.createOrder = createOrder;
    }

    public int getDestroyOrder() {
        return destroyOrder;
    }

    public void setDestroyOrder(int destroyOrder) {
        this.destroyOrder = destroyOrder;
    }

    public int getModifyOrder() {
        return modifyOrder;
    }

    public void setModifyOrder(int modifyOrder) {
        this.modifyOrder = modifyOrder;
    }

    public int getNoopOrder() {
        return noopOrder;
    }

    public void setNoopOrder(int noopOrder) {
        this.noopOrder = noopOrder;
    }

    public String getCreateVerb() {
        return createVerb;
    }

    public void setCreateVerb(String createVerb) {
        this.createVerb = createVerb;
    }

    public String getModifyVerb() {
        return modifyVerb;
    }

    public void setModifyVerb(String modifyVerb) {
        this.modifyVerb = modifyVerb;
    }

    public String getDestroyVerb() {
        return destroyVerb;
    }

    public void setDestroyVerb(String destroyVerb) {
        this.destroyVerb = destroyVerb;
    }

    public String getNoopVerb() {
        return noopVerb;
    }

    public void setNoopVerb(String noopVerb) {
        this.noopVerb = noopVerb;
    }

    public String getInspectScript() {
        return inspectScript;
    }

    public void setInspectScript(String inspectScript) {
        this.inspectScript = inspectScript;
    }
}
