package com.xebialabs.deployit.plugin.jbossdm.step;

import java.io.*;
import java.util.Timer;
import java.util.TimerTask;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.execution.ExecutionContextListener;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereProcess;
import com.xebialabs.overthere.RuntimeIOException;

public class CliDaemon implements ExecutionContextListener, Serializable {

    private String cliExecutable;
    private final String username;
    private final String password;
    private final String adminHost;
    private final int adminPort;
    private final Host connectingHost;

    private transient OverthereConnection connection;
    private transient OverthereProcess process;

    private static final int FLUSH_DELAY_MS = 5000;
    private static final int FLUSH_CHECK_INTERVAL_MS = 2000;



    public CliDaemon(String cliExecutable, String username, String password, String adminHost, int adminPort, Host connectingHost) {
        this.cliExecutable = cliExecutable;
        this.username = username;
        this.password = password;
        this.adminHost = adminHost;
        this.adminPort = adminPort;
        this.connectingHost = connectingHost;
    }

    public Object executeCliCommandWithoutDaemon(ExecutionContext context, String cmd) {
        if (connection == null) {
            connection = connectingHost.getConnection();
        } else if (!isConnected()) {
            connect(context);
        }

        CmdLine cmdLine = createBasicCliCmdLine();
        cmdLine.addArgument("--command=" + cmd);
        CliProcessOutputHandler handler = new CliProcessOutputHandler();
        int rc = connection.execute(handler, cmdLine);
        if (rc != 0) {
            throw new RuntimeIOException("Cli command failed with return code " + rc + ". Details : " + handler.getResult());
        }
        return handler.getResult();
    }

    public Object executeCliCommand(ExecutionContext context, String cmd) {
        if (!isConnected()) {
            connect(context);
        }

        CliProcessOutputHandler handler = new CliProcessOutputHandler();
        executeCommand(handler, cmd);
        return handler.getResult();
    }

    private boolean isConnected() {
        return connection != null && process != null;
    }

    public void disconnect() {
        if (process != null) {
            process.destroy();
            process = null;
        }

        if (connection != null) {
            connection.close();
            connection = null;
        }
    }

    public void connect(ExecutionContext context) {
        disconnect();
        connection = connectingHost.getConnection();
        CmdLine cmdLine = createBasicCliCmdLine();

        process = connection.startProcess(cmdLine);
        waitForCliStart(context);
    }

    private CmdLine createBasicCliCmdLine() {
        CmdLine cmdLine = new CmdLine();
        cmdLine.addArgument(cliExecutable).addArgument("--connect").addArgument("--controller=" + adminHost + ":" + adminPort);
        if (!Strings.nullToEmpty(username).trim().isEmpty()) {
            cmdLine.addArgument("--user=" + username);
        }
        if (!Strings.nullToEmpty(password).trim().isEmpty()) {
            cmdLine.addPassword("--password=" + password);
        }
        return cmdLine;
    }

    protected void waitForCliStart(final ExecutionContext context) {
        final InputStreamReader stdout = new InputStreamReader(process.getStdout());
        final StringBuilder stdoutLineBuffer = new StringBuilder();
        final long[] flushAfter = new long[1];
        final TimerTask flushTimerTask = new TimerTask() {
            @Override
            public void run() {
                synchronized (stdoutLineBuffer) {
                    if (flushAfter[0] < System.currentTimeMillis()) {
                        if (stdoutLineBuffer.length() > 0) {
                            context.logOutput(stdoutLineBuffer.toString());
                            stdoutLineBuffer.setLength(0);
                        }
                        flushAfter[0] = System.currentTimeMillis() + FLUSH_DELAY_MS;
                    }
                }
            }
        };

        new Timer("CliDaemon-AutoFlushTimer", true).schedule(flushTimerTask, FLUSH_DELAY_MS, FLUSH_CHECK_INTERVAL_MS);

        try {
            for (; ; ) {
                try {
                    int cInt = stdout.read();   //TODO : Use something like SimpleTimerLimiter from guava to fail after 10 seconds or so.
                    if (cInt == -1) {
                        captureStderr(context, process);
                        int exitCode = -1;
                        try {
                            exitCode = process.waitFor();
                        } catch (InterruptedException exc) {
                            logger.error("Interrupted while waiting for " + process + " to complete");
                            Thread.currentThread().interrupt();
                        }
                        throw new RuntimeIOException("Cannot start cli: exit code " + exitCode);
                    } else {
                        char c = (char) cInt;
                        if (c != '\r' && c != '\n') {
                            synchronized (stdoutLineBuffer) {
                                stdoutLineBuffer.append(c);
                            }
                        }
                        if (c == ']') {
                            synchronized (stdoutLineBuffer) {
                                flushAfter[0] = System.currentTimeMillis() + FLUSH_DELAY_MS;
                                String stdoutLine = stdoutLineBuffer.toString();
                                stdoutLineBuffer.setLength(0);
                                if (stdoutLine.trim().matches("\\[(domain|standalone)@.+:[0-9]+./\\]")) {
                                    break;
                                }
                                context.logOutput(stdoutLine);
                            }
                        }
                    }
                } catch (IOException exc) {
                    throw new RuntimeIOException("Cannot start cli", exc);
                }
            }
        } finally {
            flushTimerTask.cancel();
        }
    }

    private static void captureStderr(final ExecutionContext context, OverthereProcess daemonProcess) throws IOException {
        final BufferedReader stderr = new BufferedReader(new InputStreamReader(daemonProcess.getStderr()));
        for (; ; ) {
            final String stderrLine = stderr.readLine();
            if (stderrLine == null) {
                break;
            }
            context.logError(stderrLine);
        }
    }

    private void executeCommand(CliProcessOutputHandler handler, String command) {
        try {
            logger.info("Executing command {} on {} (with cli daemon)", command, connection);
            if (command.charAt(command.length() - 1) == '\n')
                command = command.substring(0, command.length() - 1);

            String daemonLine = command + "\n";
            OutputStream stdin = process.getStdin();
            stdin.write(daemonLine.getBytes());
            stdin.flush();

            InputStreamReader stdout = new InputStreamReader(process.getStdout());
            StringBuilder lineBuffer = new StringBuilder();
            for (; ; ) {
                char c = (char) stdout.read();
                if(c == -1) {
                    throw new EOFException("Unexpected end of file");
                }
                if (c == '\r') {
                    continue;
                }
                String line = lineBuffer.toString();
                if (c != '\n') {
                    lineBuffer.append(c);
                } else {
                    if (!line.equals(command)) {
                        handler.handleOutputLine(line);

                    }
                    lineBuffer = new StringBuilder();
                    continue;
                }

                if (line.trim().matches("\\[(domain|standalone)@.+:[0-9]+\\s.*\\]")) {
                    //command complete
                    break;
                } else {
                    handler.handleOutput(c);
                }
            }
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot execute command " + command + " on " + connectingHost, exc);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(CliDaemon.class);

    @Override
    public void contextDestroyed() {
        disconnect();
    }
}
