package com.xebialabs.deployit.plugin.lb.util;

import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import com.google.common.collect.ListMultimap;

import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.lb.ci.LoadBalancer;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.plugin.lb.util.DeploymentGroups.getDeploymentGroup;
import static java.lang.Boolean.TRUE;
import static java.util.Collections.sort;

public class LoadBalancedContainers {
    public static final String WEBSERVER_FRONTED_CONTAINERS_PROPERTY = "applicationServers";
    public static final String CONTAINER_REMOVE_FROM_POOL_PROPERTY = "disableInLoadBalancer";

    public static ListMultimap<Container, LoadBalancer> getContainerLoadBalancers(Set<LoadBalancer> loadBalancers) {
        ListMultimap<Container, LoadBalancer> loadBalancedContainers = ArrayListMultimap.create();

        for (LoadBalancer loadBalancer : loadBalancers) {
            for (Container appserver : loadBalancer.getManagedServers()) {
                if (TRUE.equals(appserver.getProperty(CONTAINER_REMOVE_FROM_POOL_PROPERTY))) {
                    loadBalancedContainers.put(appserver, loadBalancer);
                }
            }
        }
        return loadBalancedContainers;
    }

    /**
     * container -> first/last deployment groups of the fronted servers
     *
     * Deployment groups are identified by the 'deploymentGroup' property which is sorted numerically. So if the
     * deployment groups are
     *
     * 3 - 6 - 2
     *
     * the first group is 2 and the last group is 6. Containers without tags are in the special Integer.MAX_VALUE
     * group that is processed last.
     *
     * @param containers The containers to map.
     * @return The map of container to bounds
     */
    public static Map<Container, LoadBalancingBounds> getLoadBalancingBounds(Set<Container> containers) {
        Builder<Container, LoadBalancingBounds> loadBalancingBounds = ImmutableMap.builder();
        for (Container container : containers) {
            loadBalancingBounds.put(container, new LoadBalancingBounds(container));
        }
        return loadBalancingBounds.build();
    }

    public static class LoadBalancingBounds {

        public final int startGroup;
        public final int endGroup;

        private LoadBalancingBounds(Container container) {
            /*
             * For webservers, the load balancing bounds are determined as follows:
             * 
             * 1) Get the deployment group for each fronted server, putting servers without a group in the 'unallocated'
             * group 2) Sort all the groups found 3) The first group is the one before which the webserver needs to be
             * taken out of the load balancer pool; it can be returned after the last group has been processed
             */
            if (container.hasProperty(WEBSERVER_FRONTED_CONTAINERS_PROPERTY)) {
                List<Integer> deploymentGroups = newArrayList(getDeploymentGroup(container));
                Set<com.xebialabs.deployit.plugin.api.udm.Container> frontedServers = container.<Set<com.xebialabs.deployit.plugin.api.udm.Container>>getProperty(WEBSERVER_FRONTED_CONTAINERS_PROPERTY);
                for (com.xebialabs.deployit.plugin.api.udm.Container frontedServer : frontedServers) {
                    deploymentGroups.add(getDeploymentGroup(frontedServer));
                }
                sort(deploymentGroups);
                startGroup = deploymentGroups.get(0);
                endGroup = deploymentGroups.get(deploymentGroups.size() - 1);
            } else {
                /*
                 * For appservers, the load balancing bounds are just the deployment group of the server.
                 */
                startGroup = getDeploymentGroup(container);
                endGroup = startGroup;
            }
            logger.debug("Determined load balancing bounds for '{}': start '{}', end '{}'", container, startGroup, endGroup);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(LoadBalancedContainers.class);
}
