#
import re

from com.xebialabs.deployit.plugin.api.reflect import Type
from policy.modules.deployment.registry import PolicyRegistry
from policy.modules.util import Struct
from com.xebialabs.deployit.repository import ItemInUseException

class PackageCleaner():
    def __init__(self, context):
        self.context = Struct(context)
        self.repository_service = self.context.repositoryService
        self.placeholder_repository = self.context.placeholderRepository
        self.logger = self.context.logger

        exec "from policy.modules.deployment.policies import *"
        exec "from policy.modules.deployment.ext import *"

    def _read_package_ids(self, app_id):
        return [package.id for package in self.repository_service.query(Type.valueOf("udm.Version"), app_id, None, None, None, None, 0, -1)]

    def _pick_by_regex(self, pattern, package_ids):
        regex_pattern = re.compile(pattern)
        return [package_id for package_id in package_ids if regex_pattern.match(package_id)]

    def _find_packages_to_remove(self, app_id, job):
        package_ids = self._read_package_ids(app_id)
        package_ids = self._pick_by_regex(job.pattern, package_ids)
        if package_ids:
            policy = PolicyRegistry.new_policy(job.policyType, self.context)
            package_ids = policy.filter(app_id, job, package_ids)
        return package_ids

    def _find_and_omit_deployed_packages(self, app_id, package_ids, deployed_package_ids):
        deployed_package_ids.extend(self._get_deployed_package_ids(app_id))
        if package_ids:
            package_ids = [package_id for package_id in package_ids if package_id not in deployed_package_ids]
        return package_ids

    def _get_deployed_package_ids(self, app_id):
        deployed_apps = self.repository_service.query(Type.valueOf("udm.DeployedApplication"), None, "Environments", self._get_name(app_id), None, None, 0, -1)
        deployed_package_ids = []
        for deployed_app in deployed_apps:
            deployed_package_ids.extend(self._collect_package_ids(self.repository_service.read(deployed_app.id).version))
        return deployed_package_ids

    def _collect_package_ids(self, package):
        if package.type.instanceOf(Type.valueOf("udm.CompositePackage")):
            package_ids = [package.id]
            for nested_package in self.repository_service.read(package.id).packages:
                package_ids.extend(self._collect_package_ids(nested_package))
            return package_ids
        else:
            return [package.id]

    def _remove_packages(self, package_ids):
        self.logger.info("Removing packages...")
        skipped_packages = []
        try:
            self.repository_service.deleteList(package_ids)
        except ItemInUseException:
            for package_id in package_ids:
                try:
                    self.repository_service.delete(package_id)
                except:
                    skipped_packages.append(package_id)
        except:
            self.logger.error("Error deleting packages")
            raise
        return skipped_packages

    def _get_name(self, id):
        return id[id.rfind("/") + 1:]

    def _sort_apps_by_composite_first(self, app_id):
        return len(self.repository_service.query(Type.valueOf("udm.CompositePackage"), app_id, None, None, None, None, 0, 1)) == 0

    def run(self, job):
        self.logger.info("=== Running package purge job [%s] (retention: %s, dryRun: %s) ===" % (job.name, str(job.packageRetention), job.dryRun))
        deployed_package_ids = []
        app_ids = [app.id for app in self.repository_service.query(Type.valueOf("udm.Application"), None, "Applications", None, None, None, 0, -1)]
        app_ids = sorted(app_ids, key=self._sort_apps_by_composite_first)
        for app_id in app_ids:
            package_ids = self._find_packages_to_remove(app_id, job)
            package_ids = self._find_and_omit_deployed_packages(app_id, package_ids, deployed_package_ids)
            self.logger.info("== %s [packages to remove: %s]" % (app_id, len(package_ids)))
            if package_ids:
                self.logger.info("Packages: %s" % ", ".join([self._get_name(package_id) for package_id in package_ids]))
                if not job.dryRun:
                    skipped_packages = self._remove_packages(package_ids)
                    if skipped_packages:
                        self.logger.info("Skipped packages: %s" % ", ".join([self._get_name(skipped_package) for skipped_package in skipped_packages]))
        self.logger.info("Removing stale placeholders...")
        deleted_count = self.placeholder_repository.deleteStalePlaceholders()
        self.logger.info("Removed stale placeholders count: %s" % deleted_count)
        self.logger.info("=== Finished package purge job [%s] ===" % job.name)

