package com.xebialabs.deployit.plugin.powershell;

import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.google.common.io.Closeables;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.io.Closeables.closeQuietly;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.MDC_KEY_SCRIPT_PATH;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.dumpScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScriptDir;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.uploadScript;
import static com.xebialabs.overthere.OperatingSystemFamily.WINDOWS;
import static com.xebialabs.overthere.util.OverthereUtils.getName;

@SuppressWarnings("serial")
abstract class PowerShellStep implements Step {

    private static final String DEFAULT_POWER_SHELL_PATH = "powershell";
    private static final String STANDARD_RUNTIME_PATH = "powershell/runtime";
    private HostContainer container;
    private String scriptPath;
    private Map<String, Object> powerShellVars;
    private String description;
    private boolean uploadArtifactData = true;

    public PowerShellStep(HostContainer container, String scriptPath, Map<String, Object> powerShellVars, String description) {
        this.container = checkNotNull(container, "container is null");
        checkArgument(this.container.getHost().getOs() == WINDOWS, "PowerShell scripts can only be run on Windows hosts");
        this.scriptPath = checkNotNull(scriptPath, "scriptPath is null");
        this.powerShellVars = checkNotNull(powerShellVars, "powerShellVars is null");
        this.description = checkNotNull(description, "description is null");
    }

    protected StepExitCode doExecute(ExecutionContext ctx) throws Exception {
        MDC.put(MDC_KEY_SCRIPT_PATH, scriptPath);
        try {
            OverthereConnection conn = container.getHost().getConnection();
            try {
                String finalScript = aggregateScript(conn);
                dumpScript(getName(scriptPath), finalScript, scriptsLogger);
                OverthereFile uploadedScriptFile = uploadScript(conn, scriptPath, finalScript);

                int res = executePowerShellScript(ctx, conn, uploadedScriptFile);

                logger.debug("Exit code: {}", res);
                if(res == 0) {
                    return StepExitCode.SUCCESS;
                } else {
                    return StepExitCode.FAIL;
                }
            } finally {
                Closeables.closeQuietly(conn);
            }
        } finally {
            MDC.remove(MDC_KEY_SCRIPT_PATH);
        }
    }

    protected String aggregateScript(OverthereConnection connection) {
        String convertedVars = PowerShellVarsConverter.javaToPowerShell(connection, powerShellVars, uploadArtifactData);

        StringBuilder b = new StringBuilder();
        b.append(loadScriptDir(STANDARD_RUNTIME_PATH));
        if(container instanceof PowerShellContainer) {
            b.append(loadScriptDir(((PowerShellContainer) container).getRuntimePath()));
        }
        b.append("# PowerShellVars\n");
        b.append(convertedVars);
        b.append(loadScript(scriptPath));

        return b.toString().replaceAll("\n", "\r\n");
    }

    private int executePowerShellScript(ExecutionContext ctx, OverthereConnection conn, OverthereFile script) {
        logger.info("Executing PowerShell script {} on {}", script, conn);
        DefaultExecutionOutputHandler stdoutHandler = handleStdout(ctx);
        DefaultExecutionOutputHandler stderrHandler = handleStderr(ctx);
        try {
            CmdLine cmdLine;
            if(container instanceof PowerShellContainer) {
                cmdLine = ((PowerShellContainer) container).getScriptCommandLine(script);
            } else {
                cmdLine = getDefaultScriptCommandLine(DEFAULT_POWER_SHELL_PATH, script);
            }
            return conn.execute(stdoutHandler, stderrHandler, cmdLine);
        } finally {
            closeQuietly(stdoutHandler);
            closeQuietly(stderrHandler);
        }
    }

    @Override
    public String getDescription() {
        return description;
    }

    public void setUploadArtifactData(boolean uploadArtifactData) {
        this.uploadArtifactData = uploadArtifactData;
    }

    public static CmdLine getDefaultScriptCommandLine(String powerShellPath, OverthereFile script) {
        return CmdLine.build(powerShellPath, "-ExecutionPolicy", "Unrestricted", "-Inputformat", "None", "-NonInteractive", "-NoProfile",
            "-Command", "$ErrorActionPreference = 'Stop'; & " + script.getPath() + "; if($LastExitCode) { Exit $LastExitCode; }");
    }

    private static Logger logger = LoggerFactory.getLogger(PowerShellStep.class);

    private static final Logger scriptsLogger = LoggerFactory.getLogger("com.xebialabs.deployit.plugin.powershell.scripts");

}
