package com.xebialabs.deployit.plugin.powershell;

import java.util.Set;

import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.planning.Noop;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.validation.Regex;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.CREATE;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.DESTROY;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.NOOP;

@SuppressWarnings("serial")
@Metadata(virtual = true)
public class BaseExtensiblePowerShellDeployed<D extends Deployable> extends BasePowerShellDeployed<D, HostContainer> {

    @Property(hidden = true, defaultValue = "60", description = "Order at which the create step will be executed.")
    private int createOrder;

    @Property(hidden = true, required = false, description = "PowerShell script invoked to create a resource (or deploy an artifact) on Windows.")
    private String createScript;

    @Property(hidden = true, defaultValue = "Create", description = "Verb that is used to prefix the description for the create step.")
    private String createVerb;

    @Property(hidden = true, defaultValue = STEP_OPTION_UPLOAD_ARTIFACT_DATA, description = "Options for the create step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    private Set<String> createOptions;

    @Property(hidden = true, defaultValue = "60", description = "Order at which the modify step will be executed.")
    private int modifyOrder;

    @Property(hidden = true, required = false, description = "PowerShell script invoked to modify a resource (or an artifact) on Windows.")
    private String modifyScript;

    @Property(hidden = true, defaultValue = "Modify",description = "Verb that is used to prefix the description for the modify step.")
    private String modifyVerb;

    @Property(hidden = true, defaultValue = STEP_OPTION_UPLOAD_ARTIFACT_DATA, description = "Options for the modify step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    private Set<String> modifyOptions;

    @Property(hidden = true, defaultValue = "40", description = "Order at which the destroy step will be executed.")
    private int destroyOrder;

    @Property(hidden = true, required = false, description = "PowerShell script invoked to destroy a resource (or undeploy an artifact) on Windows.")
    private String destroyScript;

    @Property(hidden = true, defaultValue = "Destroy", description = "Verb that is used to prefix the description for the destroy step.")
    private String destroyVerb;

    @Property(hidden = true, defaultValue = STEP_OPTION_NONE, description = "Options for the destroy step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    private Set<String> destroyOptions;

    @Property(hidden = true, defaultValue = "50", description = "Order at which the noop step will be executed.")
    private int noopOrder;

    @Property(hidden = true, required = false, description = "PowerShell script invoked to noop a resource or an artifact on Windows.")
    private String noopScript;

    @Property(hidden = true, defaultValue = "Noop", description = "Verb that is used to prefix the description for the noop step.")
    private String noopVerb;

    @Property(hidden = true, defaultValue = STEP_OPTION_NONE, description = "Options for the noop step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    private Set<String> noopOptions;

    @Property(hidden = true, defaultValue = "90", description = "Order at which the start step will be executed.")
    private int startOrder;

    @Property(hidden = true, required = false, description = "PowerShell script invoked to start a resource of an artifact on Windows.")
    private String startScript;

    @Property(hidden = true, defaultValue = "Start", description = "Verb that is used to prefix the description for the start step.")
    private String startVerb;

    @Property(hidden = true, defaultValue = STEP_OPTION_NONE, description = "Options for the start step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    private Set<String> startOptions;

    @Property(hidden = true, defaultValue = "10", description = "Order at which the stop script will be executed.")
    private int stopOrder;

    @Property(hidden = true, required = false, description = "PowerShell script invoked to stop a resource or an artifact on Windows.")
    private String stopScript;

    @Property(hidden = true, defaultValue = "Stop", description = "Verb that is used to prefix the description for the stop step.")
    private String stopVerb;

    @Property(hidden = true, defaultValue = STEP_OPTION_NONE, description = "Options for the stop step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    private Set<String> stopOptions;

    @Property(hidden = true, required = false, defaultValue = "false", category = "Deployment", label = "Stop and start on every deployment",
        description = "If set to true, the stop and start scripts will be executed even if this deployable was not changed.")
    private boolean stopStartOnNoop;

    @Create
    public void create(DeploymentPlanningContext ctx, Delta delta) {
        addStepWithCheckpoint(ctx, delta, CREATE, createOrder, createScript, createVerb, createOptions);
        start(ctx, delta);
    }

    @Modify
    public void modify(DeploymentPlanningContext ctx, Delta delta) {
        if (addStepWithCheckpoint(ctx, delta, MODIFY, modifyOrder, modifyScript, modifyVerb, modifyOptions)) {
            start(ctx, delta);
            stop(ctx, delta);
        } else {
            BaseExtensiblePowerShellDeployed<?> previous = (BaseExtensiblePowerShellDeployed<?>) delta.getPrevious();
            previous.destroy(ctx, delta);

            BaseExtensiblePowerShellDeployed<?> current = (BaseExtensiblePowerShellDeployed<?>) delta.getDeployed();
            current.create(ctx, delta);
        }
    }

    @Destroy
    public void destroy(DeploymentPlanningContext ctx, Delta delta) {
        addStepWithCheckpoint(ctx, delta, DESTROY, destroyOrder, destroyScript, destroyVerb, destroyOptions);
        stop(ctx, delta);
    }

    @Noop
    public void noop(DeploymentPlanningContext ctx, Delta delta) {
        addStepWithCheckpoint(ctx, delta, NOOP, noopOrder, noopScript, noopVerb, noopOptions);
        if(stopStartOnNoop) {
            stop(ctx, delta);
            start(ctx, delta);
        }
    }

    private void start(DeploymentPlanningContext ctx, Delta delta) {
        addStep(ctx, delta, startOrder, startScript, startVerb, startOptions);
    }

    private void stop(DeploymentPlanningContext ctx, Delta delta) {
        addStep(ctx, delta, stopOrder, stopScript, stopVerb, stopOptions);
    }

}
