package com.xebialabs.deployit.plugin.powershell;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Collections2;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.remoting.vars.VarsConverter;
import com.xebialabs.overthere.OverthereConnection;

import static com.google.common.collect.Lists.newArrayList;
import static org.apache.commons.codec.binary.Base64.encodeBase64String;

public class PowerShellVarsConverter extends VarsConverter {

    public static String javaToPowerShell(OverthereConnection connection, Map<String, Object> powershellVars, boolean maskPasswords) {
        return javaToPowerShell(connection, powershellVars, true, null, maskPasswords);
    }

    public static String javaToPowerShell(OverthereConnection connection, Map<String, Object> powershellVars, boolean uploadArtifactData, ExecutionContext ctx, boolean maskPasswords) {
        List<String> powerShellList = javaToPowerShellList(connection, powershellVars, uploadArtifactData, ctx, maskPasswords);
        return Joiner.on("\n").join(powerShellList) + "\n";
    }

    static List<String> javaToPowerShellList(OverthereConnection connection, Map<String, Object> powershellVars, boolean uploadArtifactData, ExecutionContext ctx, boolean maskPasswords) {
        PowerShellVarsConverter converter = new PowerShellVarsConverter(connection, powershellVars, maskPasswords);
        converter.setUploadArtifactData(uploadArtifactData);
        converter.setExecutionContext(ctx);
        return converter.convert();
    }

    private PowerShellVarsConverter(OverthereConnection connection, Map<String, Object> pythonVars, boolean maskPasswords) {
        super(connection, pythonVars, maskPasswords);
    }

    @Override
    protected void setNullVariable(String variableName) {
        add("$" + variableName + " = $null");
    }

    @Override
    protected void setBooleanVariable(String variableName, boolean variableValue) {
        add("$" + variableName + " = " + toPowerShellBoolean(variableValue));
    }

    @Override
    protected void setIntegerVariable(String variableName, int variableValue) {
        add("$" + variableName + " = " + variableValue);
    }

    @Override
    protected void setLongVariable(String variableName, long variableValue) {
        add("$" + variableName + " = " + variableValue);
    }

    @Override
    protected void setStringVariable(String variableName, String variableValue) {
        add("$" + variableName + " = " + toPowerShellString(variableValue));
    }

    @Override
    protected void setCollectionOfStringsVariable(String variableName, Collection<?> variableValue) {
        add("$" + variableName + " = " + toPowerShellCollectionOfStrings(variableValue));
    }

    @Override
    protected void startCreateObject(String variableName) {
        add("$" + variableName + " = New-Object Object");
    }

    @Override
    protected void endCreateObject(String variableName) {
        // no-op
    }

    private void setNoteProperty(String objectName, String propertyName, String literalPropertyValue) {
        add("$" + objectName + " | Add-Member NoteProperty " + toPowerShellString(propertyName) + " " + literalPropertyValue);
        
    }

    @Override
    protected void setNullProperty(String objectName, String propertyName) {
        setNoteProperty(objectName, propertyName, "$null");
    }

    @Override
    protected void setEmptyCollectionProperty(String objectName, String propertyName) {
        setNoteProperty(objectName, propertyName, "@()");
    }

    @Override
    protected void setEmptyMapProperty(String objectName, String propertyName) {
        setNoteProperty(objectName, propertyName, "@{}");
    }

    @Override
    protected void setBooleanProperty(String objectName, String propertyName, boolean propertyValue) {
        setNoteProperty(objectName, propertyName, toPowerShellBoolean(propertyValue));
    }

    @Override
    protected void setIntegerProperty(String objectName, String propertyName, int propertyValue) {
        setNoteProperty(objectName, propertyName, Integer.toString(propertyValue));
    }

    @Override
    protected void setStringProperty(String objectName, String propertyName, String propertyValue) {
        setNoteProperty(objectName, propertyName, toPowerShellString(propertyValue));
    }

    @Override
    protected void setPasswordProperty(String objectName, String propertyName, String propertyValue) {
        setNoteProperty(objectName, propertyName, "([System.Text.Encoding]::UTF8.GetString([System.Convert]::FromBase64String(" + toPowerShellString(encodeBase64String(propertyValue.getBytes())) + ")))");
    }

    @Override
    protected void setCollectionOfStringsProperty(String objectName, String propertyName, Collection<?> propertyValue) {
        setNoteProperty(objectName, propertyName, toPowerShellCollectionOfStrings(propertyValue));
    }

    @Override
    protected void setCiReferenceProperty(String objectName, String propertyName, ConfigurationItem propertyValue) {
        setNoteProperty(objectName, propertyName, "$" + getConfigurationItemVariableName(propertyValue));
    }

    @Override
    protected void setCollectionOfCiReferencesProperty(String objectName, String propertyName, Collection<ConfigurationItem> propertyValue) {
        List<String> varRefs = newArrayList();
        for (ConfigurationItem setItem : propertyValue) {
            varRefs.add("$" + getConfigurationItemVariableName(setItem));
        }
        setNoteProperty(objectName, propertyName, "@(" + Joiner.on(", ").join(varRefs) + ")");
    }

    @Override
    protected void setMapOfStringToStringReferencesProperty(String objectName, String propertyName, Map<String, String> propertyValue) {
        List<String> entries = newArrayList();
        for (String key : propertyValue.keySet()) {
            String value = propertyValue.get(key);
            if (value == null) {
                entries.add(toPowerShellString(key) + " = $null");
            } else {
                entries.add(toPowerShellString(key) + " = " + toPowerShellString(value));
            }
        }
        setNoteProperty(objectName, propertyName, "@{" + Joiner.on("; ").join(entries) + "}");
    }

    @Override
    protected void createObjectAndSetObjectProperty(String objectName, String propertyName) {
        String embeddedObjectVariableName = generateUniqueVariableName();
        startCreateObject(embeddedObjectVariableName);
        endCreateObject(embeddedObjectVariableName);
        setNoteProperty(objectName, propertyName, "$" + embeddedObjectVariableName);
    }

    public static String toPowerShellBoolean(Boolean bool) {
        return bool ? "$true" : "$false";
    }

    public static String toPowerShellString(String str) {
        StringBuilder converted = new StringBuilder();
        converted.append("\'");
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            switch (c) {
            case '\'':
                converted.append("\'\'");
                break;
            default:
                converted.append(c);
            }
        }
        converted.append("\'");
        return converted.toString();
    }

    public static String toPowerShellCollectionOfStrings(Collection<?> variableValue) {
        Collection<String> encodedStrings = Collections2.transform(variableValue, new Function<Object, String>() {
            public String apply(Object input) {
                return toPowerShellString(input.toString());
            }
        });
        return "@(" + Joiner.on(", ").join(encodedStrings) + ")";
    }

}
