package com.xebialabs.deployit.plugin.python;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Maps.newIdentityHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.collect.Sets.newTreeSet;
import static org.apache.commons.codec.binary.Base64.encodeBase64String;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Collections2;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.ReflectionsHolder;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

public class PythonVarsConverter {

	public static final String EMBEDDED_OBJECT_INDICATOR = "_";

	public static String javaToPython(OverthereConnection connection, Map<String, Object> pythonVars) {
		List<String> pythonList = javaToPythonList(connection, pythonVars);
		return Joiner.on("\n").join(pythonList) + "\n";
	}

	static List<String> javaToPythonList(OverthereConnection connection, Map<String, Object> pythonVars) {
		return new PythonVarsConverterContext(connection, pythonVars).toPythonList();
	}

	private static class PythonVarsConverterContext {
		private static final String PYTHON_PROPERTIES_VARIABLE_NAME = "._properties";
		public static final String GENERATED_VARIABLE_PREFIX = "_pv";
		private OverthereConnection connection;
		Map<String, Object> pythonVars;

		List<String> python = newArrayList();

		Map<Object, String> alreadyConvertedCis = newIdentityHashMap();
		Map<String, String> propertyFullPathMap = newHashMap();
		int nextVarNumber = 1;

		PythonVarsConverterContext(OverthereConnection connection, Map<String, Object> pythonVars) {
			this.connection = connection;
			this.pythonVars = pythonVars;
		}

		List<String> toPythonList() {
			TreeSet<String> sortedKeys = newTreeSet(pythonVars.keySet());
			for (String varName : sortedKeys) {
				Object varValue = pythonVars.get(varName);
				toPython(varName, varValue);
			}
			return python;
		}

		private void toPython(String varName, Object varValue) {
			if (varValue == null) {
				appendNullVar(varName);
			} else if (varValue instanceof Boolean) {
				appendBooleanVar(varName, (Boolean) varValue);
			} else if (varValue instanceof Integer) {
				appendIntegerVar(varName, (Integer) varValue);
			} else if (varValue instanceof Long) {
				appendLongVar(varName, (Long) varValue);
			} else if (varValue instanceof ConfigurationItem) {
				alreadyConvertedCis.put(varValue, varName);
				appendConfigurationItemVar(varName, (ConfigurationItem) varValue);
			} else if (varValue instanceof Collection) {
				appendCollectionVar(varName, (Collection<?>) varValue);
			} else {
				appendStringVar(varName, varValue.toString());
			}
		}

		void appendConfigurationItemVar(String varName, ConfigurationItem item) {
			Descriptor d = DescriptorRegistry.getDescriptor(item.getType());
			if (d == null) {
				throw new IllegalArgumentException("Python var " + varName + " is a ConfigurationItem of an unknown type: " + item.getType());
			}

			python.add(varName + " = DictionaryObject()");
			appendEmbeddedObjectDefinitions(varName, d);
			makePropertyFullPathsMap(varName, d);
			appendConfigurationItemVarMetaData(varName, item, d);
			appendRegularPropertiesToConfigurationItemVar(varName, item, d);
			appendDerivedPropertiesToConfigurationItemVar(varName, item);
			appendArtifactPropertiesToConfigurationItemVar(varName, item);
		}

		private void appendEmbeddedObjectDefinitions(String rootObjectName, Descriptor descriptor) {
			renderEmbeddedObjects(extractEmbeddedObjectsAndAttributes(rootObjectName, descriptor));
		}

		private void renderEmbeddedObjects(
				Multimap<String, String> objectAttributesKeyedByObjectName) {
			for (String absoluteObjectName : objectAttributesKeyedByObjectName.keySet()) {
				python.add(absoluteObjectName + " = DictionaryObject()");
				StringBuilder embeddedObjectProperties = new StringBuilder();
				for(String p : objectAttributesKeyedByObjectName.get(absoluteObjectName)) {
					if(embeddedObjectProperties.length() > 0) {
						embeddedObjectProperties.append(", ");
					}
					embeddedObjectProperties.append(toPythonString(p));
				}
				python.add(absoluteObjectName + PYTHON_PROPERTIES_VARIABLE_NAME + " = [" + embeddedObjectProperties.toString() + "]");
			}
		}

		private Multimap<String, String> extractEmbeddedObjectsAndAttributes(String rootObjectName, Descriptor descriptor) {
			Multimap<String, String> objectAttributesKeyedByObjectName = ArrayListMultimap.create();
			for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
				if (isEmbeddedObjectStylePropertyName(pd.getName())) {
					
					String absoluteObjectName = rootObjectName + "." + pd.getName().replaceAll(EMBEDDED_OBJECT_INDICATOR, ".");
					String attributeName = absoluteObjectName.substring(absoluteObjectName.lastIndexOf(".") + 1);
					absoluteObjectName = absoluteObjectName.substring(0, absoluteObjectName.lastIndexOf("."));

					objectAttributesKeyedByObjectName.put(absoluteObjectName, attributeName);
				}
			}

			// Add embedded objects as properties of their parent object.
			for (String absoluteObjectName : objectAttributesKeyedByObjectName.keySet()) {
				String absoluteParentObjectName = absoluteObjectName.substring(0, absoluteObjectName.lastIndexOf("."));
				String objectName = absoluteObjectName.substring(absoluteObjectName.lastIndexOf(".") + 1);
				if (objectAttributesKeyedByObjectName.containsKey(absoluteParentObjectName)) {
					objectAttributesKeyedByObjectName.get(absoluteParentObjectName).add(objectName);
				}
			}

			return objectAttributesKeyedByObjectName;
		}

		private boolean isEmbeddedObjectStylePropertyName(String propName) {
			return propName.contains(EMBEDDED_OBJECT_INDICATOR);
		}

		private void makePropertyFullPathsMap(String varName, Descriptor descriptor) {
			propertyFullPathMap.put(varName + ".id", varName + ".id");
			propertyFullPathMap.put(varName + ".name", varName + ".name");
			for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
				String propName = varName + "." + pd.getName();
				String propertyName = pd.getName().replaceAll("_", ".");
				String propertyFullPath = varName + "." + propertyName;
				if (!propertyFullPathMap.containsKey(propName)) {
					propertyFullPathMap.put(propName, propertyFullPath);
				}
			}
		}

		private void appendConfigurationItemVarMetaData(String varName, ConfigurationItem item, Descriptor d) {
	        appendStringVar(varName + ".id", item.getId());
			appendStringVar(varName + ".name", item.getName());

			Set<String> attributes = newHashSet("id", "name");
			for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
				if(!isEmbeddedObjectStylePropertyName(pd.getName())) {
					attributes.add(pd.getName());
				} else {
					attributes.add(pd.getName().substring(0, pd.getName().indexOf(EMBEDDED_OBJECT_INDICATOR)));
				}
			}
			
			Set<Method> methodsAnnotatedWith = ReflectionsHolder.getMethodsAnnotatedWith(DerivedProperty.class);
			Descriptor descriptor = DescriptorRegistry.getDescriptor(item.getType());
			for (Method method : methodsAnnotatedWith) {
				if (descriptor.isAssignableTo(method.getDeclaringClass())) {
					DerivedProperty derivedProperty = method.getAnnotation(DerivedProperty.class);
					attributes.add(derivedProperty.value());
				}
			}

			StringBuilder properties = new StringBuilder();
			for (String attribute : attributes) {
				if (properties.length() > 0) {
					properties.append(", ");
				}
				properties.append(toPythonString(attribute));
			}
			
			appendRawVar(varName + PYTHON_PROPERTIES_VARIABLE_NAME, "[" + properties.toString() + "]");
        }

		@SuppressWarnings("unchecked")
        private void appendRegularPropertiesToConfigurationItemVar(String varName, ConfigurationItem item, Descriptor d) {
	        for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
				String propertyName = varName + "." + pd.getName();
				Object propertyValue = pd.get(item);

				switch (pd.getKind()) {
					case BOOLEAN:
						propertyName = propertyFullPathMap.get(propertyName);
						if (propertyValue == null) {
							appendNullVar(propertyName);
						} else if (!(propertyValue instanceof Boolean)) {
							throw new IllegalStateException("Property " + propertyName + " is not a Boolean but a " + propertyValue.getClass().getName());
						} else {
							appendBooleanVar(propertyName, (Boolean) propertyValue);
						}
						break;
					case INTEGER:
						propertyName = propertyFullPathMap.get(propertyName);
						if (propertyValue == null) {
							appendNullVar(propertyName);
						} else if (!(propertyValue instanceof Integer)) {
							throw new IllegalStateException("Property " + propertyName + " is not an Integer but a " + propertyValue.getClass().getName());
						} else {
							appendIntegerVar(propertyName, (Integer) propertyValue);
						}
						break;
					case SET_OF_STRING:
						if (propertyValue == null) {
							appendEmptySetVar(propertyName);
						} else  if (!(propertyValue instanceof Set)) {
							throw new IllegalStateException("Property " + propertyName + " is not a Set but a " + propertyValue.getClass().getName());
						} else {
							appendCollectionVar(propertyName, (Set<String>) propertyValue);
						}
						break;
					case CI:
						if (d.isAssignableTo(Type.valueOf(Deployed.class)) && pd.getName().equals(Deployed.DEPLOYABLE_FIELD)) {
							appendNullVar(propertyName);
						} else if (propertyValue == null) {
							appendNullVar(propertyName);
						} else if (!(propertyValue instanceof ConfigurationItem)) {
							throw new IllegalStateException("Property " + propertyName + " is not a ConfigurationItem but a " + propertyValue.getClass().getName());
						} else {
							appendCiReferenceVar(propertyName, (ConfigurationItem) propertyValue);
						}
						break;
					case SET_OF_CI:
						if (propertyValue == null) {
							appendEmptySetVar(propertyName);
						} else if (!(propertyValue instanceof Set)) {
							throw new IllegalStateException("Property " + propertyName + " is not a Set but a " + propertyValue.getClass().getName());
						} else {
							appendSetOfCiReferencesVar(propertyName, (Set<ConfigurationItem>) propertyValue);
						}
						break;
                    case MAP_STRING_STRING:
                    	propertyName = propertyFullPathMap.get(propertyName);
						if (propertyValue == null) {
							appendEmptyMapVar(propertyName);
						} else if (!(propertyValue instanceof Map)) {
							throw new IllegalStateException("Property " + propertyName + " is not a Map but a " + propertyValue.getClass().getName());
						} else {
							appendMapOfStringsReferencesVar(propertyName, (Map<String, String>) propertyValue);
						}
						break;
					case STRING:
					case ENUM:
						propertyName = propertyFullPathMap.get(propertyName);
						if (propertyValue == null) {
							appendNullVar(propertyName);
						} else if (pd.isPassword()) {
							appendPasswordVar(propertyName, propertyValue.toString());
						} else {
							appendStringVar(propertyName, propertyValue.toString());
						}
						break;

					default:
					    throw new IllegalStateException("Should not end up here!");
				}
			}
        }

        void appendDerivedPropertiesToConfigurationItemVar(String varName, ConfigurationItem item) {
			Set<Method> methodsAnnotatedWith = ReflectionsHolder.getMethodsAnnotatedWith(DerivedProperty.class);
			Descriptor descriptor = DescriptorRegistry.getDescriptor(item.getType());
			for (Method method : methodsAnnotatedWith) {
				if (descriptor.isAssignableTo(method.getDeclaringClass())) {
					DerivedProperty derivedProperty = method.getAnnotation(DerivedProperty.class);
					String propertyName = varName + "." + derivedProperty.value();
					try {
						toPython(propertyName, method.invoke(item));
					} catch (IllegalAccessException e) {
						throw new RuntimeException("Method " + method.getName() + " in class " + method.getClass() + " cannot be accessed.", e);
					} catch (InvocationTargetException e) {
						throw new RuntimeException("Method " + method.getName() + " in class " + method.getClass() + " cannot be invoked.", e);
					}
				}
			}
		}

		private void appendArtifactPropertiesToConfigurationItemVar(String varName, ConfigurationItem item) {
	        if (item instanceof Artifact) {
				Artifact a = (Artifact) item;
				checkNotNull(a.getFile(), a + " has a null file");
				OverthereFile uploadedFileArtifact = connection.getTempFile(a.getFile().getName());
				a.getFile().copyTo(uploadedFileArtifact);
				appendStringVar(varName + ".file", uploadedFileArtifact.getPath());
			}
        }

		void appendNullVar(String varName) {
			python.add(varName + " = None");
		}

		private void appendEmptySetVar(String varName) {
			python.add(varName + " = []");
        }

        private void appendEmptyMapVar(String varName) {
			python.add(varName + " = {}");
        }

		void appendBooleanVar(String varName, Boolean varValue) {
			python.add(varName + " = " + (varValue ? "True" : "False"));
			python.add(varName + "_as_string = " + (varValue ? "\'true\'" : "'false\'"));
		}

		void appendIntegerVar(String varName, Integer varValue) {
			python.add(varName + " = " + varValue);
		}

		void appendLongVar(String varName, Long varValue) {
			python.add(varName + " = " + varValue + "L");
		}

		void appendCollectionVar(String varName, Collection<?> varValue) {
			if (varValue.isEmpty()) {
				python.add(varName + " = []");
			} else {
				Collection<String> encodedStrings = Collections2.transform(varValue, new Function<Object, String>() {
	                public String apply(Object input) {
		                return toPythonString(input.toString());
	                } } );
				python.add(varName + " = [" + Joiner.on(", ").join(encodedStrings) + "]");
			}
		}

		void appendStringVar(String varName, String varValue) {
			python.add(varName + " = " + toPythonString(varValue));
		}

		void appendPasswordVar(String varName, String varValue) {
			python.add(varName + " = base64.decodestring(" + toPythonString(encodeBase64String(varValue.getBytes())) + ")");
		}

		void appendRawVar(String varName, String varValue) {
			python.add(varName + " = " + varValue);
		}

		void appendCiReferenceVar(String varName, ConfigurationItem varValue) {
			python.add(varName + " = " + appendNewConfigurationItemVar(varValue));
		}

		void appendSetOfCiReferencesVar(String varName, Set<ConfigurationItem> varValue) {
			if (varValue.isEmpty()) {
				appendEmptySetVar(varName);
			} else {
				List<String> varRefs = newArrayList();
				for (ConfigurationItem setItem : varValue) {
					varRefs.add(appendNewConfigurationItemVar(setItem));
				}
				python.add(varName + " = [" + Joiner.on(", ").join(varRefs) + "]");
			}
		}

        void appendMapOfStringsReferencesVar(String varName, Map<String, String> varValue) {
            if (varValue.isEmpty()) {
                appendEmptyMapVar(varName);
            } else {
                List<String> varRefs = newArrayList();
                for (String key : varValue.keySet()) {
                    String value = varValue.get(key);
                    if (value == null) {
                        varRefs.add("" + toPythonString(key) + ": None");
                    } else {
                        varRefs.add("" + toPythonString(key) + ": " + toPythonString(value));
                    }
                }
                python.add(varName + " = {" + Joiner.on(", ").join(varRefs) + "}");
            }
        }


        String appendNewConfigurationItemVar(ConfigurationItem item) {
			if (alreadyConvertedCis.containsKey(item)) {
				return alreadyConvertedCis.get(item);
			}

			String pvName = GENERATED_VARIABLE_PREFIX + nextVarNumber++;
			alreadyConvertedCis.put(item, pvName);

			appendConfigurationItemVar(pvName, item);
			return pvName;
		}
	}

	public static String toPythonString(String str) {
		StringBuilder converted = new StringBuilder();
		converted.append("\"");
		for (int i = 0; i < str.length(); i++) {
			char c = str.charAt(i);
			switch (c) {
			case '\r':
				converted.append("\\r");
				break;
			case '\n':
				converted.append("\\n");
				break;
			case '\\':
			case '\'':
			case '\"':
				converted.append('\\');
				// deliberate fall-through
			default:
				converted.append(c);
			}
		}
		converted.append("\"");
		return converted.toString();
	}

}
