package com.xebialabs.deployit.plugin.python;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.CONNECT_FROM_DAEMON;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.CONNECT_FROM_STAND_ALONE_SCRIPT;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.DISCONNECT_FROM_DAEMON;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.DISCONNECT_FROM_STAND_ALONE_SCRIPT;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.RUN_SCRIPT_FROM_DAEMON;
import static com.xebialabs.deployit.plugin.python.PythonVarsConverter.toPythonString;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.MDC_KEY_SCRIPT_PATH;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.dumpScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScriptDir;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.uploadScript;
import static com.xebialabs.overthere.util.OverthereUtils.getName;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.google.common.io.Closeables;
import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.ExecutionContextListener;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.overthere.ExecutionContextOverthereProcessOutputHandler;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.OverthereProcess;
import com.xebialabs.overthere.RuntimeIOException;

@SuppressWarnings("serial")
abstract class PythonStep<C extends ExecutionContext> implements Step<C> {

	private static final String STANDARD_RUNTIME_PATH = "python/runtime";
	private static final String DAEMON_SCRIPT = "python/daemon/daemon.py";
	private static final String DAEMON_STARTED = "DEPLOYIT-DAEMON-STARTED";
	private static final String DAEMON_EXIT_CODE_MARKER = "DEPLOYIT-DAEMON-EXIT-VALUE: ";
	private static final String DAEMON_END_OF_COMMAND_MARKER = "DEPLOYIT-DAEMON-END-OF-COMMAND";

	private final PythonManagingContainer container;
	private final String scriptPath;
	private final Map<String, Object> pythonVars;
	private final String description;
    private boolean uploadArtifactData = true;

	public PythonStep(PythonManagingContainer container, String scriptPath, Map<String, Object> pythonVars, String description) {
		this.container = checkNotNull(container, "container is null");
		this.scriptPath = checkNotNull(scriptPath, "scriptPath is null");
		this.pythonVars = checkNotNull(pythonVars, "pythonVars is null");
		this.description = checkNotNull(description, "description is null");
	}

	protected Result doExecute(ExecutionContext ctx) throws Exception {
		MDC.put(MDC_KEY_SCRIPT_PATH, scriptPath);
		try {
			OverthereConnection conn = container.getHost().getConnection();
			try {
				String finalScript = aggregateScript(conn);
				dumpScript(getName(scriptPath), finalScript, logger);
				OverthereFile uploadedScriptFile = uploadScript(conn, scriptPath, finalScript);
	
				int res = executePythonScript(ctx, conn, uploadedScriptFile);
	
				return (res == 0) ? Result.Success : Result.Fail;
			} finally {
				Closeables.closeQuietly(conn);
			}
		} finally {
			MDC.remove(MDC_KEY_SCRIPT_PATH);
		} 
	}

	protected String aggregateScript(OverthereConnection connection) {
		String pythonVarsPython = PythonVarsConverter.javaToPython(connection, pythonVars, uploadArtifactData);
		StringBuilder b = new StringBuilder();
        if(!container.runWithDaemon() || !connection.canStartProcess()) {
            b.append(loadScriptDir(STANDARD_RUNTIME_PATH));
            //for backward compatibility
            b.append(loadScriptDir(container.getRuntimePath()));
            //for synthetic extensibility
            if (container.hasProperty("runtimePaths")) {
                List<String> paths = container.getProperty("runtimePaths");
                for(String path: paths) {
                    b.append(loadScriptDir(path));
                }
            }
        }
		b.append("# PythonVars\n");
		b.append(pythonVarsPython);
		if(!container.runWithDaemon() || !connection.canStartProcess()) {
			b.append("#\n" + CONNECT_FROM_STAND_ALONE_SCRIPT + "()\n");
		}
		b.append(loadScript(scriptPath));
		if(!container.runWithDaemon() || !connection.canStartProcess()) {
			b.append("#\n" + DISCONNECT_FROM_STAND_ALONE_SCRIPT + "()\n");
		}
        return b.toString();
	}

    private int executePythonScript(final ExecutionContext ctx, final OverthereConnection conn, final OverthereFile script) {
		if (container.runWithDaemon() && conn.canStartProcess()) {
			return executePythonScriptWithDaemon(ctx, script);
		} else {
			return conn.execute(new ExecutionContextOverthereProcessOutputHandler(ctx), container.getScriptCommandLine(script));
		}
	}

    private int executePythonScriptWithDaemon(final ExecutionContext ctx, final OverthereFile script) {
		try {
			DaemonProcessWrapper daemonProcessWrapper = getDaemonProcess(ctx);
			OverthereProcess daemonProcess = daemonProcessWrapper.getProcess();
			logger.info("Executing Python script {} on {}", script, daemonProcessWrapper.getConnection());

			String daemonLine = RUN_SCRIPT_FROM_DAEMON + "(" + toPythonString(script.getPath()) + ")\n";
			OutputStream stdin = daemonProcess.getStdin();
			stdin.write(daemonLine.getBytes());
			stdin.flush();

			final BufferedReader stdout = new BufferedReader(new InputStreamReader(daemonProcess.getStdout()));
			final BufferedReader stderr = new BufferedReader(new InputStreamReader(daemonProcess.getStderr()));
			Thread stderrThread = new Thread(new Runnable() {
				public void run() {
					try {
						while (!Thread.interrupted()) {
							String stderrLine = stderr.readLine();
							if (stderrLine == null) {
								break;
							}
							ctx.logError(stderrLine);
						}
					} catch (InterruptedIOException ie) {
						// ignore, is expected
					} catch (IOException exc) {
						exc.printStackTrace();
					}
				}
			}, "stderr printer for python daemon running on " + this);
			stderrThread.start();
			try {
				int exitCode = 0;
				for (;;) {
					String stdoutLine = stdout.readLine();
					if (stdoutLine == null) {
						throw new RuntimeIOException("Cannot execute script " + script.getPath() + " on " + container.getHost() + ": lost connection to the python daemon");
					}

					if (stdoutLine.startsWith(DAEMON_EXIT_CODE_MARKER)) {
						try {
							exitCode = Integer.parseInt(stdoutLine.substring(DAEMON_EXIT_CODE_MARKER.length()));
						} catch (NumberFormatException ignored) {
						}
					} else if (stdoutLine.equals(DAEMON_END_OF_COMMAND_MARKER)) {
						break;
					} else {
						ctx.logOutput(stdoutLine);
					}
				}

				return exitCode;
			} finally {
				stderrThread.interrupt();
			}
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot execute script " + script.getPath() + " on " + container.getHost(), exc);
		}
	}

	private DaemonProcessWrapper getDaemonProcess(final ExecutionContext context) {
		String key = "DAEMON_" + container.getId();
		DaemonProcessWrapper wrappedProcess = (DaemonProcessWrapper) context.getAttribute(key);
		if (wrappedProcess == null) {
            context.logOutput("Starting daemon on " + container.getHost());
			OverthereConnection daemonConnection = container.getHost().getConnection();

			String daemonScript = generateDaemonScript();
			dumpScript("daemon.py", daemonScript, logger);
			OverthereFile uploadedDaemonScript = uploadScript(daemonConnection, "daemon.py", daemonScript);

			OverthereProcess daemonProcess = daemonConnection.startProcess(container.getScriptCommandLine(uploadedDaemonScript));
			final BufferedReader stdout = new BufferedReader(new InputStreamReader(daemonProcess.getStdout()));
			for (;;) {
				String stdoutLine;
				try {
					stdoutLine = stdout.readLine();
					if (stdoutLine == null) {
						throw new RuntimeIOException("Cannot start python daemon: lost connection to the python daemon");
					}
					context.logOutput(stdoutLine);
					if (stdoutLine.startsWith(DAEMON_STARTED)) {
						break;
					}
				} catch (IOException exc) {
					throw new RuntimeIOException("Cannot start python daemon", exc);
				}
			}

			wrappedProcess = new DaemonProcessWrapper(daemonConnection, daemonProcess);
			context.setAttribute(key, wrappedProcess);
		}
		return wrappedProcess;
	}

	private String generateDaemonScript() {
	    StringBuilder b = new StringBuilder();
		b.append(loadScriptDir(STANDARD_RUNTIME_PATH));
		//for backward compatibility
        b.append(loadScriptDir(container.getRuntimePath()));
        //for synthetic extensibility
        if (container.hasProperty("runtimePaths")) {
            List<String> paths = container.getProperty("runtimePaths");
            for(String path: paths) {
                b.append(loadScriptDir(path));
            }
        }
		b.append("#\n" + CONNECT_FROM_DAEMON + "()\n");
	    b.append(loadScript(DAEMON_SCRIPT));
		b.append("#\n" + DISCONNECT_FROM_DAEMON + "()\n");
	    return b.toString();
    }

	private static final class DaemonProcessWrapper implements ExecutionContextListener {

		private final OverthereConnection connection;

		private final OverthereProcess process;

		private DaemonProcessWrapper(final OverthereConnection connection, final OverthereProcess process) {
			this.connection = connection;
			this.process = process;
		}

		private OverthereConnection getConnection() {
			return connection;
		}

		private OverthereProcess getProcess() {
			return process;
		}

		public void contextDestroyed() {
			OutputStream stdin = process.getStdin();
			try {
				stdin.write("QUIT\r\n".getBytes());
				stdin.flush();
				process.waitFor();
			} catch (IOException exc) {
				logger.error("Error stopping python daemon", exc);
			} catch (RuntimeException exc) {
				logger.error("Error stopping python daemon", exc);
			} catch (InterruptedException exc) {
				logger.error("Error stopping python daemon", exc);
			} finally {
				Closeables.closeQuietly(connection);
			}

		}
	}

	String getScriptPath() {
		return scriptPath;
	}

	Map<String, Object> getPythonVars() {
		return pythonVars;
	}

    public void setUploadArtifactData(boolean uploadArtifactData) {
        this.uploadArtifactData = uploadArtifactData;
    }

	@Override
	public String getDescription() {
		return description;
	}

	@Override
	public String toString() {
		return PythonStep.class.getName() + "{" + "scriptName='" + scriptPath + "'}";
	}

	private static final Logger logger = LoggerFactory.getLogger(PythonStep.class.getName());

}
