package com.xebialabs.deployit.plugin.python;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Collections2;

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.remoting.vars.VarsConverter;
import com.xebialabs.overthere.OverthereConnection;

import static com.google.common.collect.Lists.newArrayList;
import static org.apache.commons.codec.binary.Base64.encodeBase64String;

public class PythonVarsConverter extends VarsConverter {

    private static final String INDENT = "    ";

    private int objectDepth = 0;

    private int currentFunctionNumber = 0;
    
    private boolean shouldOpenFunction = false;

    private boolean functionOpened = false;

    public static String javaToPython(OverthereConnection connection, Map<String, Object> pythonVars) {
        return javaToPython(connection, pythonVars, true);
    }

    public static String javaToPython(OverthereConnection connection, Map<String, Object> pythonVars, boolean uploadArtifactData) {
        List<String> pythonList = javaToPythonList(connection, pythonVars,uploadArtifactData);
        return Joiner.on("\n").join(pythonList) + "\n";
    }

    static List<String> javaToPythonList(OverthereConnection connection, Map<String, Object> pythonVars, boolean uploadArtifactData) {
        PythonVarsConverter converter = new PythonVarsConverter(connection, pythonVars);
        converter.setUploadArtifactData(uploadArtifactData);
        return converter.convert();
    }

    private PythonVarsConverter(OverthereConnection connection, Map<String, Object> pythonVars) {
        super(connection, pythonVars, DerivedProperty.class);
    }

    @Override
    protected void setNullVariable(String varName) {
        add(varName + " = None");
    }

    @Override
    protected void setBooleanVariable(String varName, boolean varValue) {
        add(varName + " = " + (varValue ? "True" : "False"));
        add(varName + "_as_string = " + (varValue ? "\'true\'" : "'false\'"));
    }

    @Override
    protected void setIntegerVariable(String varName, int varValue) {
        add(varName + " = " + varValue);
    }

    @Override
    protected void setLongVariable(String varName, long varValue) {
        add(varName + " = " + varValue + "L");
    }

    @Override
    protected void setStringVariable(String varName, String varValue) {
        add(varName + " = " + toPythonString(varValue));
    }

    @Override
    protected void setCollectionOfStringsVariable(String varName, Collection<?> varValue) {
        if (varValue.isEmpty()) {
            add(varName + " = []");
        } else {
            Collection<String> encodedStrings = Collections2.transform(varValue, new Function<Object, String>() {
                public String apply(Object input) {
                    return toPythonString(input.toString());
                }
            });
            add(varName + " = [" + Joiner.on(", ").join(encodedStrings) + "]");
        }
    }

    @Override
    protected void startCreateObject(String varName) {
        if(objectDepth++ > 0) {
            endFunction();
        }
        startFunction();
        if(varName.indexOf('.') == -1) {
            add(INDENT + "global " + varName);
        }
        add(INDENT + varName + " = DictionaryObject()");
    }

    @Override
    protected void endCreateObject(String varName) {
        endFunction();
        if(--objectDepth > 0) {
            startFunction();
        }
    }

    private void startFunction() {
        shouldOpenFunction = true;
        functionOpened = false;
    }

    private void endFunction() {
        if(functionOpened) {
            super.add("_vars_" + currentFunctionNumber + "()");
            functionOpened = false;
        }
    }

    @Override
    public void add(String line) {
        if(shouldOpenFunction && !functionOpened) {
            super.add("def _vars_" + ++currentFunctionNumber + "():");
            shouldOpenFunction = false;
            functionOpened = true;
        }
        super.add(line);
    }

    @Override
    protected void setNullProperty(String objectName, String propertyName) {
        setNullVariable(INDENT + objectName + "." + propertyName);
    }

    @Override
    protected void setEmptyCollectionProperty(String objectName, String propertyName) {
        add(INDENT + objectName + "." + propertyName + " = []");
    }

    @Override
    protected void setEmptyMapProperty(String objectName, String propertyName) {
        add(INDENT + objectName + "." + propertyName + " = {}");
    }

    @Override
    protected void setBooleanProperty(String objectName, String propertyName, boolean varValue) {
        setBooleanVariable(INDENT + objectName + "." + propertyName, varValue);
    }

    @Override
    protected void setIntegerProperty(String objectName, String propertyName, int varValue) {
        setIntegerVariable(INDENT + objectName + "." + propertyName, varValue);
    }

    @Override
    protected void setStringProperty(String objectName, String propertyName, String varValue) {
        setStringVariable(INDENT + objectName + "." + propertyName, varValue);
    }

    @Override
    protected void setPasswordProperty(String objectName, String propertyName, String varValue) {
        add(INDENT + objectName + "." + propertyName + " = base64.decodestring(" + toPythonString(encodeBase64String(varValue.getBytes())) + ")");
    }

    @Override
    protected void setCollectionOfStringsProperty(String objectName, String propertyName, Collection<?> varValue) {
        setCollectionOfStringsVariable(INDENT + objectName + "." + propertyName, varValue);
    }

    @Override
    protected void setCiReferenceProperty(String objectName, String propertyName, ConfigurationItem varValue) {
        add(INDENT + objectName + "." + propertyName + " = " + getConfigurationItemVariableName(varValue));
    }

    @Override
    protected void setCollectionOfCiReferencesProperty(String objectName, String propertyName, Collection<ConfigurationItem> varValue) {
        if (varValue.isEmpty()) {
            add(INDENT + objectName + "." + propertyName + " = []");
        } else {
            List<String> referencedVarNames = newArrayList();
            for (ConfigurationItem setItem : varValue) {
                referencedVarNames.add(getConfigurationItemVariableName(setItem));
            }
            add(INDENT + objectName + "." + propertyName + " = [" + Joiner.on(", ").join(referencedVarNames) + "]");
        }
    }

    @Override
    protected void setMapOfStringToStringReferencesProperty(String objectName, String propertyName, Map<String, String> varValue) {
        if (varValue.isEmpty()) {
            setEmptyMapProperty(objectName, propertyName);
        } else {
            List<String> entries = newArrayList();
            for (String key : varValue.keySet()) {
                String value = varValue.get(key);
                if (value == null) {
                    entries.add(toPythonString(key) + ": None");
                } else {
                    entries.add(toPythonString(key) + ": " + toPythonString(value));
                }
            }
            add(INDENT + objectName + "." + propertyName + " = {" + Joiner.on(", ").join(entries) + "}");
        }
    }

    protected void createObjectAndSetObjectProperty(String objectName, String propertyName) {
        String embeddedObjectVariableName = objectName + "." + propertyName;
        startCreateObject(embeddedObjectVariableName);
        endCreateObject(embeddedObjectVariableName);
    }

    public static String toPythonString(String str) {
        StringBuilder converted = new StringBuilder();
        converted.append("\"");
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            switch (c) {
            case '\r':
                converted.append("\\r");
                break;
            case '\n':
                converted.append("\\n");
                break;
            case '\\':
            case '\'':
            case '\"':
                converted.append('\\');
                // deliberate fall-through
            default:
                converted.append(c);
            }
        }
        converted.append("\"");
        return converted.toString();
    }

}
