package com.xebialabs.xldeploy.authentication.oidc.conf

import ai.digital.config.ServerConfigurationHelper
import ai.digital.configuration.central.deploy.{ClientProperties, ServerSideProperties}
import com.xebialabs.deployit.core.auth.LoginMetadataService
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.plumbing.authentication.WithoutRedirectLoginSuccessHandler
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.deployit.security.authentication.{BasicAuthWithRememberMeFilter, RememberMeAuthenticationProvider}
import com.xebialabs.deployit.taskexecution.security.{TaskWorkerAuthenticationFilter, TaskWorkerAuthenticationProvider}
import com.xebialabs.deployit.{LicenseExpiryCheckFilter, LogbackAccessSecurityAttributesSaveFilter}
import com.xebialabs.license.LicenseValidationFilter
import com.xebialabs.license.service.LicenseService
import com.xebialabs.platform.sso.oidc.service.XLOidcUserService
import com.xebialabs.platform.sso.oidc.web.{CustomAuthorizationRequestResolver, OidcLogoutSuccessHandler}
import com.xebialabs.xldeploy.auth.BasicAuthOverridingHttpSessionSecurityContextRepository
import com.xebialabs.xldeploy.auth.config.InMemoryConfigurer
import com.xebialabs.xldeploy.auth.oidc.web.XlDeployLoginFormFilter
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.{Autowired, Qualifier, Value}
import org.springframework.context.ApplicationContext
import org.springframework.context.annotation.Lazy
import org.springframework.http.HttpMethod
import org.springframework.security.access.AccessDecisionManager
import org.springframework.security.authentication.{AuthenticationManager, AuthenticationProvider}
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder
import org.springframework.security.config.annotation.web.builders.{HttpSecurity, WebSecurity}
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter
import org.springframework.security.config.http.SessionCreationPolicy
import org.springframework.security.core.session.SessionRegistry
import org.springframework.security.ldap.authentication.LdapAuthenticationProvider
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService
import org.springframework.security.oauth2.client.endpoint.DefaultAuthorizationCodeTokenResponseClient
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository
import org.springframework.security.oauth2.client.web.OAuth2AuthorizedClientRepository
import org.springframework.security.oauth2.jwt.JwtDecoder
import org.springframework.security.web.access.AccessDeniedHandler
import org.springframework.security.web.access.channel.ChannelProcessingFilter
import org.springframework.security.web.authentication.logout.HttpStatusReturningLogoutSuccessHandler
import org.springframework.security.web.authentication.session.{ChangeSessionIdAuthenticationStrategy, CompositeSessionAuthenticationStrategy, ConcurrentSessionControlAuthenticationStrategy, SessionAuthenticationStrategy}
import org.springframework.security.web.authentication.{AuthenticationFailureHandler, DelegatingAuthenticationEntryPoint, RememberMeServices, UsernamePasswordAuthenticationFilter}
import org.springframework.security.web.context.request.async.WebAsyncManagerIntegrationFilter
import org.springframework.security.web.csrf.{CsrfAuthenticationStrategy, HttpSessionCsrfTokenRepository}
import org.springframework.security.web.firewall.StrictHttpFirewall
import org.springframework.security.web.savedrequest.NullRequestCache
import org.springframework.web.filter.RequestContextFilter

import java.util
import java.util.{LinkedList => LList, List => JList}
import scala.jdk.CollectionConverters.ListHasAsScala

abstract class DeploySecurityConfig extends WebSecurityConfigurerAdapter {

  private val logger = LoggerFactory.getLogger(classOf[DeploySecurityConfig])

  @Value("${deploy.server.license.days-before-warning:5}")
  var daysBeforeWarning: Int = _

  @Autowired
  var openIdLogoutSuccessHandler: OidcLogoutSuccessHandler = _

  @Autowired
  var applicationContext: ApplicationContext = _

  @Autowired
  var tokenRepository: HttpSessionCsrfTokenRepository = _

  @Autowired
  var loginMetadataService: LoginMetadataService = _

  @Autowired
  var roleService: RoleService = _

  @Autowired
  var accessDeniedHandler: AccessDeniedHandler = _

  @Autowired
  @Qualifier("rememberMeKey")
  var rememberMeKey: String = _

  @Autowired
  var rememberMeServices: RememberMeServices = _

  @Autowired
  var withoutRedirectLogoutSuccessHandler: HttpStatusReturningLogoutSuccessHandler = _

  @Autowired
  var workerRepository: TaskExecutionWorkerRepository = _

  @Autowired
  var licenseService: LicenseService = _

  @Autowired
  var taskWorkerAuthenticationProvider: TaskWorkerAuthenticationProvider = _

  @Autowired(required = false)
  var ldapAuthenticationProvider: java.util.List[LdapAuthenticationProvider] = _

  @Autowired
  var accessDecisionManager: AccessDecisionManager = _

  @Lazy
  @Autowired
  @Qualifier("xlAuthenticationProvider")
  var xlAuthenticationProvider: AuthenticationProvider = _

  @Autowired
  var delegatingAuthenticationEntryPoint: DelegatingAuthenticationEntryPoint = _

  @Autowired
  var delegatingSecurityContextRepository: BasicAuthOverridingHttpSessionSecurityContextRepository = _

  @Autowired
  var authorizedClientRepository: OAuth2AuthorizedClientRepository = _

  @Autowired
  var clientRegistrationRepository: ClientRegistrationRepository = _

  @Autowired
  var authorizedClientService: OAuth2AuthorizedClientService = _

  @Autowired
  var xlOidcUserService: XLOidcUserService = _

  @Autowired
  var authorizationCodeTokenResponseClient: DefaultAuthorizationCodeTokenResponseClient = _

  @Autowired
  var idTokenDecoderFactory: JwtDecoder = _

  @Autowired
  @Qualifier("oidcLoginFailureHandler")
  var oidcLoginFailureHandler: AuthenticationFailureHandler = _

  @Autowired
  var customAuthorizationRequestResolver: CustomAuthorizationRequestResolver = _

  @Autowired
  @Qualifier("authenticationFailureHandler")
  var authenticationFailureHandler: AuthenticationFailureHandler = _

  @Autowired
  var serverSideConfiguration: ServerSideProperties = _

  @Autowired
  var sessionRegistry: SessionRegistry = _

  @Autowired
  var clientProperties: ClientProperties = _

  override def configure(auth: AuthenticationManagerBuilder): Unit = {

    InMemoryConfigurer.configure(auth)

    auth
      .authenticationProvider(taskWorkerAuthenticationProvider)
      .authenticationProvider(new RememberMeAuthenticationProvider())
      .authenticationProvider(xlAuthenticationProvider)

    if (ldapAuthenticationProvider != null) {
      for (provider <- ldapAuthenticationProvider.asScala) {
        auth
          .authenticationProvider(provider)
      }
    }
  }

  def configureSecurity(web: WebSecurity, prefix: String): Unit = {
    val firewall = new StrictHttpFirewall()
    firewall.setAllowUrlEncodedSlash(true)
    firewall.setAllowUrlEncodedPercent(true)
    web.httpFirewall(firewall)

    web.ignoring().antMatchers("/productregistration")
    web.ignoring().antMatchers(s"/$prefix/internal/configuration/properties")
    web.ignoring().antMatchers(HttpMethod.GET, s"/$prefix/settings/general")
    web.ignoring().antMatchers(s"/$prefix/ha/health")
    if (clientProperties.ignoreAuthCheckForServerStateApi)
      web.ignoring().antMatchers(s"/$prefix/server/state")

    web.ignoring().antMatchers("/**.js**")
    web.ignoring().antMatchers("/**.html")
    web.ignoring().antMatchers("/**.css")
    web.ignoring().antMatchers("/favicon.ico")
    web.ignoring().antMatchers("/fonts/**")
    web.ignoring().antMatchers("/icons/**")
    web.ignoring().antMatchers("/images/**")
  }

  def licenseValidationFilter: LicenseValidationFilter = {
    val filter = new LicenseValidationFilter()
    filter.setLicenseService(licenseService)
    filter
  }

  private def defaultOidcMustacheTemplateSettings: util.Map[String, String] = {
    val scope = new util.HashMap[String, String]
    scope.put("productLogo", "./images/xl-deploy-logo.svg")
    scope.put("productColor", "#0099cc")
    scope.put("productIcon", "./favicon.ico")
    scope.put("productName", "XL Deploy")
    scope
  }

  protected def buildLoginFormFilter(authenticationManager: AuthenticationManager,
                                   sessionAuthenticationStrategy: SessionAuthenticationStrategy): XlDeployLoginFormFilter = {
    val filter = new XlDeployLoginFormFilter(defaultOidcMustacheTemplateSettings)
    filter.setAuthenticationManager(authenticationManager)
    filter.setSessionAuthenticationStrategy(sessionAuthenticationStrategy)
    filter.setAuthenticationFailureHandler(authenticationFailureHandler)
    filter
  }

  protected def csrfAuthenticationStrategy: CompositeSessionAuthenticationStrategy = {
    val delegateStrategies: JList[SessionAuthenticationStrategy] = new LList[SessionAuthenticationStrategy]()
    delegateStrategies.add(new ChangeSessionIdAuthenticationStrategy())
    delegateStrategies.add(new CsrfAuthenticationStrategy(tokenRepository))
    delegateStrategies.add(concurrentSessionControlAuthenticationStrategy)
    new CompositeSessionAuthenticationStrategy(delegateStrategies)
  }

  protected def noCsrfAuthenticationStrategy: CompositeSessionAuthenticationStrategy = {
    val delegateStrategies: JList[SessionAuthenticationStrategy] = new LList[SessionAuthenticationStrategy]()
    delegateStrategies.add(new ChangeSessionIdAuthenticationStrategy())
    delegateStrategies.add(concurrentSessionControlAuthenticationStrategy)
    new CompositeSessionAuthenticationStrategy(delegateStrategies)
  }

  private def concurrentSessionControlAuthenticationStrategy: ConcurrentSessionControlAuthenticationStrategy = {
    val authenticationStrategy = new ConcurrentSessionControlAuthenticationStrategy(sessionRegistry)
    authenticationStrategy.setMaximumSessions(serverSideConfiguration.session.maximumSessions)
    authenticationStrategy.setExceptionIfMaximumExceeded(serverSideConfiguration.session.exceptionIfMaximumExceeded)
    authenticationStrategy
  }

  def configureSecurity(http: HttpSecurity, prefix: String): Unit = {
    http
      .securityContext()
      .securityContextRepository(delegatingSecurityContextRepository)
      .and()
      .addFilterBefore(licenseValidationFilter, classOf[WebAsyncManagerIntegrationFilter])
      .addFilterBefore(new RequestContextFilter(), classOf[ChannelProcessingFilter])
      .addFilterAfter(new TaskWorkerAuthenticationFilter(authenticationManager, workerRepository), classOf[UsernamePasswordAuthenticationFilter])
      .addFilterBefore(new LogbackAccessSecurityAttributesSaveFilter(), classOf[TaskWorkerAuthenticationFilter])
      .addFilterAfter(new BasicAuthWithRememberMeFilter(authenticationManager, delegatingAuthenticationEntryPoint), classOf[TaskWorkerAuthenticationFilter])
      .addFilterAfter(new LicenseExpiryCheckFilter(licenseService, daysBeforeWarning), classOf[TaskWorkerAuthenticationFilter])
      .exceptionHandling()
      .authenticationEntryPoint(delegatingAuthenticationEntryPoint)
      .and()
      .formLogin()
      .failureHandler(authenticationFailureHandler)
      .successHandler(new WithoutRedirectLoginSuccessHandler())
      .loginPage("/login/external-login")
      .and()
      .rememberMe().key(rememberMeKey).rememberMeServices(rememberMeServices).and()
      .logout().logoutSuccessHandler(openIdLogoutSuccessHandler).and()

      .sessionManagement().sessionCreationPolicy(SessionCreationPolicy.NEVER).and()

      .requestCache().requestCache(new NullRequestCache()).and()
      .exceptionHandling().accessDeniedHandler(accessDeniedHandler).and()

      .authorizeRequests()
      .antMatchers(s"/$prefix/auth").permitAll()
      .antMatchers("/centralConfiguration/**").hasRole(ServerConfigurationHelper.ConfigRole)
      .antMatchers("/actuator/**").hasRole("ADMIN")
      .antMatchers("/v1/roles/**").denyAll()
      .antMatchers(s"/$prefix/**", "/ws").fullyAuthenticated()
      .accessDecisionManager(accessDecisionManager).and()

      .sessionManagement().maximumSessions(serverSideConfiguration.session.maximumSessions).maxSessionsPreventsLogin(serverSideConfiguration.session.exceptionIfMaximumExceeded)

    http
      .oauth2Login()
      .failureHandler(oidcLoginFailureHandler)
      .authorizedClientService(authorizedClientService)
      .authorizedClientRepository(authorizedClientRepository)
      .authorizationEndpoint()
        .authorizationRequestResolver(customAuthorizationRequestResolver)
        .and()
      .clientRegistrationRepository(clientRegistrationRepository)
      .loginProcessingUrl("/login/external-login")
      .userInfoEndpoint()
        .oidcUserService(xlOidcUserService)

    http
      .oauth2Client()
      .authorizedClientService(authorizedClientService)
      .authorizedClientRepository(authorizedClientRepository)
      .clientRegistrationRepository(clientRegistrationRepository)
      .authorizationCodeGrant()
      .authorizationRequestResolver(customAuthorizationRequestResolver)
      .accessTokenResponseClient(authorizationCodeTokenResponseClient)

    http
      .oauth2ResourceServer()
      .jwt()
      .decoder(idTokenDecoderFactory)

    http
      .oauth2Login()
      .tokenEndpoint()
      .accessTokenResponseClient(authorizationCodeTokenResponseClient)

    logger.info("Starting url prefix '{}' with oidc security config", prefix)
  }
}
