package com.xebialabs.xldeploy.authentication.oidc.conf

import com.xebialabs.xldeploy.auth.oidc.web.XlDeployLoginFormFilter
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.core.annotation.Order
import org.springframework.security.config.annotation.web.builders.HttpSecurity
import org.springframework.security.config.annotation.web.configuration.{EnableWebSecurity, WebSecurityCustomizer}
import org.springframework.security.web.SecurityFilterChain
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter
import org.springframework.security.web.authentication.logout.{CookieClearingLogoutHandler, LogoutFilter}
import org.springframework.security.web.csrf.CsrfLogoutHandler
import org.springframework.security.web.util.matcher.AntPathRequestMatcher.antMatcher

@ConditionalOnProperty(name = Array("deploy.server.security.auth.provider"), havingValue = "oidc")
@Configuration
@EnableWebSecurity
@Order(org.springframework.core.Ordered.LOWEST_PRECEDENCE)
class DeployNoCsrfSecurityConfig extends DeploySecurityConfig {

  private val prefix = "deployit"

  @Bean
  @throws[Exception]
  def securityFilterChainNoCsrfSecurityConfig(http: HttpSecurity): SecurityFilterChain = {
    configureSecurity(http, prefix = prefix)

    http
      .headers(hc =>
        hc
          .frameOptions(fc =>
            fc
              .sameOrigin()
          )
      )
      .authorizeHttpRequests(aur =>
        aur
          .requestMatchers(antMatcher("/api/**")).fullyAuthenticated()
      )
      .csrf(csrf =>
        csrf
          .disable()
      )
      .addFilterBefore(publicLogoutFilter, classOf[LogoutFilter])
      .addFilterBefore(defaultLoginFormFilter(), classOf[UsernamePasswordAuthenticationFilter])
      .build()
  }

  @Bean
  def webNoCsrfSecurityCustomizer: WebSecurityCustomizer = {
    web => configureSecurity(web, prefix = prefix)
  }

  private def defaultLoginFormFilter(): XlDeployLoginFormFilter =
    buildLoginFormFilter(authenticationManager, noCsrfAuthenticationStrategy)

  private def publicLogoutFilter: LogoutFilter = {
    val logoutFilter = new LogoutFilter(
      openIdLogoutSuccessHandler,
      new CsrfLogoutHandler(tokenRepository),
      new CookieClearingLogoutHandler("cookiesToClear", "XSRF-TOKEN")
    )
    logoutFilter.setFilterProcessesUrl("/xldeploy/logout")
    logoutFilter
  }

}
