/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */

package com.xebialabs.deployit.plugin.stitch.service.engine.processor.handler.groovy;

import com.fasterxml.jackson.databind.JsonNode;

import com.xebialabs.xlplatform.documentation.PublicApi;

import java.util.List;
import java.util.Map;


/**
 * Provides a set of mutation operations with a Document.
 */
@PublicApi
public interface Document {

  /**
   * Get original document.
   *
   * @return original document as {@link JsonNode}
   */
  JsonNode getDocument();

  /**
   * Read value from a document by a given json path.
   *
   * @param path  json path to read value from.
   * @param clazz a class of a value type
   * @param <T>   a type of a value
   * @return a value from a specified path
   */
  <T> T readValue(String path, Class<T> clazz);

  default Object readValue(String path) {
    return readValue(path, Object.class);
  }

  /**
   * Read a list of values from a document by a given json path.
   *
   * @param path  json path to read value from.
   * @param clazz a class of a value type
   * @param <T>   a type of a value
   * @return a value from a specified path
   */
  <T> List<T> readValues(String path, Class<T> clazz);

  default List<Object> readValues(String path) {
    return readValues(path, Object.class);
  }

  /**
   * Read a list of strings from a document by a given json path.
   *
   * @param path json path to read value from.
   * @return a value from a specified path
   */
  List<String> readPaths(String path);

  /**
   * Set property to a document by a given path.
   *
   * @param path  json path of a property
   * @param value value to set
   * @return same document instance
   */
  Document set(String path, Object value);

  /**
   * Map a specific {@link MapFunction} on a given path.
   *
   * @param path        json path where function applies
   * @param mapFunction value to set
   * @return same document instance
   */
  Document map(String path, MapFunction mapFunction);

  /**
   * Delete property from a document by a given path.
   *
   * @param path json path of a property
   * @return same document instance
   */
  Document delete(String path);

  /**
   * Add new property to a document by a given path.
   * Works for array properties.
   *
   * @param path  json path of a property
   * @param value value to set
   * @return same document instance
   */
  Document add(String path, Object value);

  /**
   * Put a key/value pair to a document by a given json path.
   *
   * @param path a json path of where new key should be added
   * @param key a key of a property
   * @param value a value of a property
   * @return same document instance
   */
  Document put(String path, String key, Object value);

  /**
   * Rename a key at a given json path.
   *
   * @param path where the old key is located
   * @param oldKeyName old key to rename
   * @param newKeyName a new key that old key should be changed to
   * @return same document instance
   */
  Document renameKey(String path, String oldKeyName, String newKeyName);

  /**
   * Map representation of Document.
   *
   * @return a map representation of a document
   */
  Map<String, Object> asMap();

  /**
   * Json representation of Document.
   *
   * @return a Json String representation of a Document
   */
  String asJson();

  /**
   * YAML representation of Document.
   * @return a Yaml String representation of a Document
   */
  String asYaml();

  /**
   * Returns a result document after all applied operations.
   *
   * @return mutated document after all applied operations
   */
  Document getMutatedDocument();
}
