package com.xebialabs.deployit.plugin.tomcat.processor;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.generic.deployed.AbstractDeployed;
import com.xebialabs.deployit.plugin.tomcat.deployed.ContextElement;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;

import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;

public class Utils {
    public static final Type CONTEXT_ELEMENT_TYPE = Type.valueOf(ContextElement.class);

    public static Iterable<AbstractDeployed<?>> findDeployedsTargetedToContainer(List<Delta> deltas, final Type containerType) {
        Iterable<Delta> destroyNoopCreateDeltasTargetedToContainer = filter(deltas,
                DeltasTargetedToContainerFilter.filterAllOperationsBesidesModifyOps(containerType));

        Iterable<Delta> modifyDestroyDeltasTargetedToContainer = filter(deltas,
                DeltasTargetedToContainerFilter.filterOnlyModifyOpsAndPreviousDeployed(containerType));

        Iterable<Delta> modifyCreateDeltasTargetedToContainer = filter(deltas,
                DeltasTargetedToContainerFilter.filterOnlyModifyOpsAndCurrentDeployed(containerType));

        Iterable<AbstractDeployed<?>> destroyNoopAndCreateDeployeds = transform(destroyNoopCreateDeltasTargetedToContainer,
                new DeltaToAbstractDeployedTransformer());

        Iterable<AbstractDeployed<?>> destroyDeployedsFromModifyDeltas = transform(modifyDestroyDeltasTargetedToContainer,
                new DeltaToAbstractDeployedTransformer(true));

        Iterable<AbstractDeployed<?>> createDeployedsFromModifyDeltas = transform(modifyCreateDeltasTargetedToContainer,
                new DeltaToAbstractDeployedTransformer());

        return Iterables.concat(destroyNoopAndCreateDeployeds, destroyDeployedsFromModifyDeltas, createDeployedsFromModifyDeltas);
    }


    public static boolean isSubtypeOf(Type supertype, Type subtype) {
        Collection<Type> typeAndSubtypes = DescriptorRegistry.getSubtypes(supertype);
        typeAndSubtypes.add(supertype);
        boolean contains = typeAndSubtypes.contains(subtype);
        logger.trace("Check {} is a subtype of {} is {} ", new Object[]{subtype, supertype, contains});
        return contains;
    }

    public static Iterable<ContextElement<?>> extractDeployedContextElements(Iterable<AbstractDeployed<?>> abstractDeployeds) {
        return transform(filter(abstractDeployeds,
                new Predicate<AbstractDeployed<?>>() {
                    @Override
                    public boolean apply(AbstractDeployed<?> input) {
                        return isSubtypeOf(CONTEXT_ELEMENT_TYPE, input.getType());
                    }
                }),
                new Function<AbstractDeployed<?>, ContextElement<?>>() {
                    @Override
                    public ContextElement<?> apply(AbstractDeployed<?> input) {
                        return (ContextElement) input;
                    }
                });
    }

    private static class DeltasTargetedToContainerFilter implements Predicate<Delta> {
        private Type type;
        private boolean modifyOpsMode;
        private boolean filterPreviousDeployed;

        DeltasTargetedToContainerFilter(Type type) {
            this.type = type;
        }

        DeltasTargetedToContainerFilter(Type type, boolean modifyOpsMode, boolean filterPreviousDeployed) {
            this.type = type;
            this.modifyOpsMode = modifyOpsMode;
            this.filterPreviousDeployed = filterPreviousDeployed;
        }

        static DeltasTargetedToContainerFilter filterAllOperationsBesidesModifyOps(Type type) {
            return new DeltasTargetedToContainerFilter(type);
        }

        static DeltasTargetedToContainerFilter filterOnlyModifyOpsAndPreviousDeployed(Type type) {
            return new DeltasTargetedToContainerFilter(type, true, true);
        }

        static DeltasTargetedToContainerFilter filterOnlyModifyOpsAndCurrentDeployed(Type type) {
            return new DeltasTargetedToContainerFilter(type, true, false);
        }

        @Override
        public boolean apply(Delta input) {
            if (input.getOperation() == Operation.MODIFY) {
                if (!modifyOpsMode) {
                    return false;
                }
                Deployed<?, ?> deployed = filterPreviousDeployed ? input.getPrevious() : input.getDeployed();
                return accept(deployed, type);
            } else {
                return accept(input.getPrevious(), type) || accept(input.getDeployed(), type);
            }
        }

        private boolean accept(Deployed<?, ?> deployed, Type type) {
            return deployed != null && isSubtypeOf(type, deployed.getContainer().getType()) &&
                    deployed instanceof AbstractDeployed;
        }
    }

    private static class DeltaToAbstractDeployedTransformer implements Function<Delta, AbstractDeployed<?>> {
        private boolean transformPreviousDeployedForModifyOp;

        DeltaToAbstractDeployedTransformer() {
        }

        DeltaToAbstractDeployedTransformer(boolean transformPreviousDeployedForModifyOp) {
            this.transformPreviousDeployedForModifyOp = transformPreviousDeployedForModifyOp;
        }

        @Override
        public AbstractDeployed<?> apply(Delta input) {
            Deployed<?, ?> deployed = input.getDeployed();

            if (input.getOperation() == Operation.MODIFY) {
                if (transformPreviousDeployedForModifyOp) {
                    deployed = input.getPrevious();
                }
            } else if (input.getOperation() == Operation.DESTROY) {
                deployed = input.getPrevious();
            }
            return (AbstractDeployed<?>) deployed;
        }
    }


    private static final Logger logger = LoggerFactory.getLogger(Utils.class);

}
