package com.xebialabs.deployit.plugin.tomcat.processor;

import java.util.Collections;
import java.util.List;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;

import com.xebialabs.deployit.plugin.api.deployment.planning.PrePlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.tomcat.deployed.ContextElement;
import com.xebialabs.deployit.plugin.tomcat.deployed.ContextWarModule;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.base.Strings.isNullOrEmpty;

/**
 * Injects the application (war) context root into all associated resources.
 * Logic:
 * <pre>
 *  Per Virtual Host :
 *       if 0 war:
 *          User would have had to manually enter context.
 *          Validate that all context elements have a context set.
 *       else if 1 war:
 *           fill/modify context for the context elements with the context from the war
 *           Validate that all context elements have a context set.
 *       else other (2+ wars):
 *           fill context for the context elements if a single war create operation in progress.
 *           for all modified/noop/destroy wars, update their associated context elements if the war context has changed.
 *           validate that all context elements have a context set.
 *  </pre>
 */
public class ApplicationContextNameInjector {

    @PrePlanProcessor
    public List<Step> injectContextNameInToContextElements(DeltaSpecification spec) {
        ListMultimap<Container, Delta> deployedsPerVirtualHost = groupByVirtualHosts(spec.getDeltas());

        for (Container virtualHost : deployedsPerVirtualHost.keySet()) {

            FluentIterable<Delta> deltasForVirtualHost = FluentIterable.from(deployedsPerVirtualHost.get(virtualHost));
            FluentIterable<Delta> warModules = deltasForVirtualHost.filter(DeltaTypePredicate.CONTEXT_WAR_MODULES);
            FluentIterable<Delta> contextElements = deltasForVirtualHost.filter(DeltaTypePredicate.CONTEXT_ELEMENTS);

            switch (warModules.size()) {
                case 0:
                    //validate will be performed below
                    break;
                case 1:
                    injectContextNameIntoContextElementsForSingleWar(warModules.get(0), contextElements);
                    break;
                default:
                    injectContextNameIntoContextElementsForMultipleWar(warModules, contextElements);
                    break;
            }

            validateAllContextElementsTargetedToVirtualHostHasCtxNameSet(contextElements);
        }

        return Collections.emptyList();
    }

    private ListMultimap<Container, Delta> groupByVirtualHosts(Iterable<Delta> deltas) {
        Iterable<Delta> deltasTargetedToVirtualHosts = findDeltasTargetedToVirtualHosts(deltas);
        return Multimaps.index(deltasTargetedToVirtualHosts, new Function<Delta, Container>() {
            @Override
            public Container apply(Delta input) {
                Deployed<?, ?> deployed = getDeployed(input);
                return deployed.getContainer();
            }
        });
    }

    private void injectContextNameIntoContextElementsForSingleWar(Delta warModuleDelta, FluentIterable<Delta> contextElements) {
        ContextWarModule<?> warModule = (ContextWarModule<?>) warModuleDelta.getDeployed();
        ContextWarModule<?> previousWarModule = (ContextWarModule<?>) warModuleDelta.getPrevious();
        for (Delta ceDelta : contextElements) {
            ContextElement<?> ce = (ContextElement<?>) getDeployed(ceDelta);
            //update context elements with war context when empty
            if (isNullOrEmpty(ce.getContext())) {
                ce.setContext(warModule.getContextRoot());
            } else if (previousWarModule != null && previousWarModule.getContextRoot().equals(ce.getContext())) {
                //update context elements context since war context root could be modified.
                warModule = warModule == null ? previousWarModule : warModule;   //incase destroy operation
                ce.setContext(warModule.getContextRoot());
            }
        }
    }

    private void injectContextNameIntoContextElementsForMultipleWar(FluentIterable<Delta> warModules, FluentIterable<Delta> contextElements) {
        FluentIterable<Delta> warModulesBeingCreated = warModules.filter(DeltaOperationPredicate.CREATE);
        if (warModulesBeingCreated.size() == 1) {
            injectContextNameIntoContextElementsForSingleWar(warModulesBeingCreated.get(0), contextElements);
        } else {
            //user must manually set the context for each context element.
            validateAllContextElementsTargetedToVirtualHostHasCtxNameSet(contextElements);
        }

        //update an context root's that may have changed since the user last created the link between context element and war.
        FluentIterable<Delta> otherWarModules = warModules.filter(Predicates.not(DeltaOperationPredicate.CREATE));
        for (Delta otherWarModule : otherWarModules) {
            injectContextNameIntoContextElementsForSingleWar(otherWarModule, contextElements);
        }
    }

    private void validateAllContextElementsTargetedToVirtualHostHasCtxNameSet(FluentIterable<Delta> contextElements) {
        for (Delta ceDelta : contextElements) {
            ContextElement<?> ce = (ContextElement<?>) getDeployed(ceDelta);
            checkState(emptyToNull(ce.getContext()) != null, "Context for resource %s targeted to %s cannot be determined. Please specify the context.", ce.getName(), ce.getContainer().getName());
        }
    }

    private Iterable<Delta> findDeltasTargetedToVirtualHosts(Iterable<Delta> deltas) {
        final Type APPLICATION_CONTEXT_TYPE = Type.valueOf("tomcat.VirtualHost");
        return FluentIterable.from(deltas).filter(new Predicate<Delta>() {
            @Override
            public boolean apply(Delta input) {
                Deployed<?, ?> deployed = getDeployed(input);
                if (deployed.getContainer().getType().getDescriptor().isAssignableTo(APPLICATION_CONTEXT_TYPE)) {
                    return deployed instanceof ContextElement || deployed instanceof ContextWarModule;
                }
                return false;
            }
        });
    }

    private Deployed<?, ?> getDeployed(Delta delta) {
        return delta.getOperation() == Operation.DESTROY ? delta.getPrevious() : delta.getDeployed();
    }


    enum DeltaTypePredicate implements Predicate<Delta> {
        CONTEXT_ELEMENTS(ContextElement.class),
        CONTEXT_WAR_MODULES(ContextWarModule.class);

        private Class<?> type;

        DeltaTypePredicate(Class<?> type) {
            this.type = type;
        }

        @Override
        public boolean apply(Delta input) {
            return type.isInstance(getDeployed(input));
        }

        private Deployed<?, ?> getDeployed(Delta delta) {
            return delta.getOperation() == Operation.DESTROY ? delta.getPrevious() : delta.getDeployed();
        }
    }

    enum DeltaOperationPredicate implements Predicate<Delta> {
        CREATE(Operation.CREATE),
        DESTROY(Operation.DESTROY),
        MODIFY(Operation.MODIFY),
        NOOP(Operation.NOOP);

        private Operation op;

        DeltaOperationPredicate(Operation op) {
            this.op = op;
        }

        @Override
        public boolean apply(Delta input) {
            return input.getOperation() == op;
        }
    }

}
