package com.xebialabs.deployit.plugin.trigger.contrib;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.engine.spi.execution.StepExecutionStateEvent;
import com.xebialabs.deployit.engine.spi.execution.TaskExecutionStateEvent;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.trigger.*;

import java.util.List;
import java.util.Map;

import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
public class TriggerExecutionListener implements ExecutionStateListener {

    private List<Trigger> allTriggers;
    private final Deltas deltas;
    private final DeployedApplication deployedApplication;

    public TriggerExecutionListener(List<Trigger> triggers, Deltas deltas, DeployedApplication deployedApplication) {
        this.allTriggers = triggers;
        this.deltas = deltas;
        this.deployedApplication = deployedApplication;
    }

    @Override
    public void stepStateChanged(final StepExecutionStateEvent event) {
        Iterable<Trigger> stepTriggers = filter(allTriggers, new FireTriggerPredicate<StepExecutionState>(StepTrigger.class, event.previousState(), event.currentState()));

        Map<String, Object> commonCtx = newHashMap();
        commonCtx.put("task", event.task());
        commonCtx.put("step", event.step());

        executeTriggers(stepTriggers, commonCtx);
    }

    protected void executeTriggers(Iterable<Trigger> triggers, Map<String, Object> commonCtx) {
        commonCtx.put("deltas", deltas);
        commonCtx.put("deployedApplication", deployedApplication);

        Iterable<Action> actions = concat(transform(triggers, new Function<Trigger, Iterable<Action>>() {
            @Override
            public Iterable<Action> apply(Trigger input) {
                return input.getActions();
            }
        }));

        for (Action action : actions) {
            Map<String, Object> ctx = newHashMap(commonCtx);
            ctx.put("action", action);
            action.execute(ctx);
        }
    }

    @Override
    public void taskStateChanged(final TaskExecutionStateEvent event) {
        Iterable<Trigger> taskTriggers = filter(allTriggers, new FireTriggerPredicate<TaskExecutionState>(TaskTrigger.class, event.previousState(), event.currentState()));
        Map<String, Object> commonCtx = newHashMap();
        commonCtx.put("task", event.task());
        executeTriggers(taskTriggers, commonCtx);
    }

    static class FireTriggerPredicate<T> implements Predicate<Trigger> {
        private Class<?> triggerType;
        private T fromState;
        private T toState;

        public FireTriggerPredicate(Class<?> triggerType, T fromState, T toState) {
            this.triggerType = triggerType;
            this.fromState = fromState;
            this.toState = toState;
        }


        @Override
        public boolean apply(Trigger input) {
            if (triggerType.isInstance(input)) {
                if (input.getFromState().equivalentTo(fromState) && input.getToState().equivalentTo(toState)) {
                    return true;
                }
            }
            return false;
        }
    }
}
