package com.xebialabs.deployit.plugin.cloud.vsphere.steps;

import java.util.List;
import java.util.concurrent.Callable;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.cloud.step.ContextAttribute;
import com.xebialabs.deployit.plugin.cloud.step.plan.Unique;
import com.xebialabs.deployit.plugin.cloud.util.Retrier;
import com.xebialabs.deployit.plugin.cloud.vsphere.access.VsphereAdapter;
import com.xebialabs.deployit.plugin.cloud.vsphere.ci.HostTemplate;

import static com.google.common.base.Preconditions.checkNotNull;

@Unique
public class WaitForVsphereIpsStep implements Step {

    private VsphereAdapter adapter;

        private HostTemplate template;

        public WaitForVsphereIpsStep(HostTemplate template, VsphereAdapter adapter) {
            this.adapter = adapter;
            this.template = template;
        }

        @Override
        public int getOrder() {
            return DEFAULT_ORDER + 15;
        }

        @Override
        public String getDescription() {
            return "Wait for all VMs to receive IPs";
        }

    @Override
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {

        List<String> vms = checkNotNull((List<String>)ctx.getAttribute(ContextAttribute.CREATED_INSTANCES.name()));

        ctx.logOutput(vms.size() + " VMs should get IP addresses.");

        if (vms.isEmpty()) {
            return StepExitCode.SUCCESS;
        }

        for (final String vm : vms) {

            ctx.logOutput("Checking IP address of " + vm);

            Boolean ok = new Retrier<Boolean>(new Callable<Boolean>() {
                @Override
                public Boolean call() throws Exception {
                    String ipAddress = adapter.getIpAddress(vm);
                    if (ipAddress != null) {
                        ctx.logOutput("Detected IP address " + ipAddress);
                        return true;
                    }

                    throw new IllegalStateException("IP address is not available yet. Waiting.");
                }
            }).retryFor(template.getBootTimeout(), template.getRetryDelay());

            if (!ok) {
                return StepExitCode.FAIL;
            }
        }


        return StepExitCode.SUCCESS;
    }
}
