package com.xebialabs.deployit.plugin.was.step;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.was.container.ManagedWebServer;
import com.xebialabs.deployit.plugin.was.container.Node;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xlplatform.satellite.Satellite;
import com.xebialabs.xlplatform.satellite.SatelliteAware;

import com.google.common.io.Closeables;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.overthere.util.OverthereFileCopier.copy;
import static java.lang.String.format;

@SuppressWarnings("serial")
public class ManagedWebServerPluginConfigurationCopyStep implements Step, SatelliteAware {
    private final int order;
    private final String description;
    private final Host srcHost;
    private final Host targetHost;
    private final String pluginSrcFilePath;
    private final String pluginDestFilePath;

    public ManagedWebServerPluginConfigurationCopyStep(int order, ManagedWebServer webserver) {
        this.order = order;
        Node webserverNode = webserver.getNode();
        srcHost = webserverNode.getCell().getCellHost();
        targetHost = webserver.getProperty("host");
        pluginSrcFilePath = format("%s/config/cells/%s/nodes/%s/servers/%s/plugin-cfg.xml",
                webserverNode.getCell().getWasHome(), webserverNode.getCellName(),
                webserverNode.getNodeName(), webserver.getName().trim());
        pluginDestFilePath = checkNotNull(webserver.<String>getProperty("pluginConfigurationPath"), "pluginConfigurationPath");
        description = format("Copy web server plugin configuration file directly to '%s' for webserver '%s'",
                targetHost.getName(), webserver.getName());
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public int getOrder() {
        return order;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        OverthereConnection srcHostConnection = srcHost.getConnection();
        OverthereConnection targetHostConnection = targetHost.getConnection();
        try {
            final OverthereFile source = srcHostConnection.getFile(pluginSrcFilePath);
            final OverthereFile remote = targetHostConnection.getFile(pluginDestFilePath);
            ctx.logOutput(format("Copying generated plugin configuration file from '%s' on '%s' to '%s' on '%s'",
                pluginSrcFilePath, srcHost, pluginDestFilePath, targetHost));
            copy(source, remote);
        } finally {
            Closeables.close(srcHostConnection, true);
            Closeables.close(targetHostConnection, true);
        }
        return StepExitCode.SUCCESS;
    }

    @Override
    public Satellite getSatellite() {
        return srcHost.getSatellite();
    }
}
