from java.util import HashSet
import re

########################################################################################################################
#
# Given an application name, will try to find the context root. Fails if cannot be resolved.
#
# @returns context root name as string.
########################################################################################################################
def getContextRoot(appName):
    appData = AdminApp.view(appName,["-CtxRootForWebMod"]).splitlines()
    for line in appData:
        if line.startswith("Context Root: ") or line.startswith("ContextRoot: "):
            return line.split(": ")[1].strip()
    print >> sys.stderr, "Cannot resolve context root for", appName
    sys.exit(1)


########################################################################################################################
#
# Given an application name, will try to find associated virtual host. Returns None if not found
#
# @returns Id of virtual host as stored in the Deployit repository.
########################################################################################################################
def getVirtualHostIdForApp(appName):
    appData = AdminApp.view(appName,["-MapWebModToVH"]).splitlines()
    for line in appData:
        if line.startswith("Virtual host: "):
            vhost = line.split(": ")[1]
            return container.id + "/" + vhost.strip()

    print "Cannot find virtual host for application ", appName
    return None

########################################################################################################################
#
# Given a shared library name, will try to find the scope in which the shared library is deployed and will resolve
# create the id as it would be stored in the Deployit repo.
# Fails if cannot resolve.
#
# @returns Id of shared library as stored in the Deployit repository.
########################################################################################################################
def getSharedLibraryId(libName):
    for c in findAllContainers(container):
        sharedLibraryContainmentPath = '%s/Library:/' % (c.containmentPath)
        sharedLibraries = AdminConfig.getid(sharedLibraryContainmentPath);
        if sharedLibraries != "":
            for sharedLibrary in sharedLibraries.splitlines():
                if libName == AdminConfig.showAttribute(sharedLibrary, 'name'):
                    libId = c.id + '/' + libName
                    return libId

    print >> sys.stderr, "Cannot resolve shared library ", libName, " in any scope"
    sys.exit(1)

########################################################################################################################
#
# Given an application name, will find all shared libraries associated with it.
#
# @returns a HashSet of String. Contains Id of shared library as stored in the Deployit repository.
########################################################################################################################
def resolveSharedLibraries(appName):
    sharedLibraries = HashSet()
    appData = AdminApp.view(appName,["-MapSharedLibForMod"]).splitlines()
    for line in appData:
        if line.startswith("Shared Libraries: ") and line.strip() != "Shared Libraries:" and line.strip() != "Shared Libraries:  null":
            #Shared Libraries:  WebSphere:name=itest-shared-library,isSharedClassloader=true+WebSphere:name=MYTestSharedList,isSharedClassloader=true
            sharedLibsLine = line.split(": ")[1].strip()
            sharedLibsLine = sharedLibsLine.split("WebSphere:name=")

            for sl in sharedLibsLine:
                if len(sl) > 0:
                    sharedLibraries.add(getSharedLibraryId(sl.split(",")[0]))
    slList = []
    for sl in sharedLibraries.toArray():
        slList.append(sl)
    return slList

########################################################################################################################
#
# Given an application name, will find all web servers associated with it.
#
# @returns a Array of String. Contains Id of webserver as stored in the Deployit repository.
########################################################################################################################
def resolveWebServers(appName):
     webServers = []
     deploymentId = AdminConfig.getid("/Deployment:%s" % (appName))
     deploymentTargets = AdminConfig.showAttribute(deploymentId, "deploymentTargets")[1:-1].splitlines()
     for dt in deploymentTargets:
        if ("nodeName" not in wsadminToDict(AdminConfig.show(dt)).keys()):
            continue # this is not a web server (most likely a cluster)

        serverContainmentPath = "/Node:%s/Server:%s/" % (AdminConfig.showAttribute(dt,"nodeName"), AdminConfig.showAttribute(dt,"name"))
        server = AdminConfig.getid(serverContainmentPath)
        if server != "":
            wasServerType = AdminConfig.showAttribute(server, 'serverType')
            if wasServerType == 'WEB_SERVER':
                if container.wasConfigIdType == 'Cell':
                    # WAS ND
                    webServers.append("%s/%s/%s" % (container.id, AdminConfig.showAttribute(dt,"nodeName"), AdminConfig.showAttribute(dt,"name")))
                else:
                    # WAS SA
                    webServers.append("%s/%s" % (container.id, AdminConfig.showAttribute(dt,"name")))
     return webServers

########################################################################################################################
#
# Given an application name, will find all security roles associated with it.
#
# @returns a Map. Key is roleName. Value is pipe separated string with groups.
########################################################################################################################
def getSecurityRoleMappings(appName):
    mappings = {}
    appData = AdminApp.view(appName,["-MapRolesToUsers"]).splitlines()
    currentRole = None
    for line in appData:
        if line.startswith("Role: "):
            currentRole = line.split(": ")[1].strip()
        if line.startswith("Mapped groups: ") and currentRole is not None:
            groups = line.split(": ")[1].strip()
            mappings[currentRole] = groups
            currentRole = None
    return mappings


def findApplicationModuleType(appName):
    descriptorFile = "%s/config/cells/%s/applications/%s.ear/deployments/%s/META-INF/application.xml" %(container.wasHome, container.cellName, appName, appName)
    appxml =  open(descriptorFile, 'r').read()
    start_string = "<display-name>"
    end_string = "</display-name>"
    start_index = appxml.find(start_string) + len(start_string)
    end_index = appxml.find(end_string)
    displayName = appxml[start_index:end_index]
    if re.match( r'.+([0-9a-f]{10,})', displayName) or displayName.endswith("_jar") or displayName.endswith("_war"):
        module = AdminApp.listModules(appName).splitlines()
        if len(module) == 1:
            if module[0].endswith("META-INF/ejb-jar.xml"):
                return "ejb"
            elif module[0].endswith("WEB-INF/web.xml"):
                return "war"

    return "ear"


########################################################################################################################
#
# Function to discover and inspect applications running on the given Deployit container (scope).
# Security role mappings and shared libraries are automatically discovered/inspected.
# Use the callback mechanism to inspect additional properties.
#
# c : The container on which applications must be discovered.
#
# canHandleAppTypeCallback :  when application is found this function is called to check if caller can handle app type.
#     appType - Can be 'war','ejb' or 'ear'
#
# appInspectCallback : if the caller wishes to inspect additional information.
#     deployedId - id of discovered ci.
#     application - application name.
#     wasDeploymentObjectId - WAS deployment object id for the application.
#     container - the deployit container on which the application was discovered.
#
########################################################################################################################
def discoverAndInspectAppModules(c, canHandleAppTypeCallback, appInspectCallback=None):
    scope = None
    if c.wasTargetType == "cluster":
        scope = "WebSphere:cell=%s,cluster=%s" %(container.name, c.name)
    else:
        scope = "WebSphere:cell=%s,node=%s,server=%s" %(container.cellName, c.node.nodeName, c.name)

    apps = AdminApp.list(scope)
    if apps == "":
        return

    for app in apps.splitlines():
        appModuleType = findApplicationModuleType(app)
        if not canHandleAppTypeCallback(appModuleType):
            print app, " %s of type %s is not a %s. Skipping." % (app, appModuleType, prototype.type)
            continue
        deployedName = app
        deployedId = c.id + '/' + deployedName
        discovered(deployedId, prototype.type)

        deploymentId = AdminConfig.getid("/Deployment:%s" % (app))
        deploymentObjectId = AdminConfig.showAttribute(deploymentId, "deployedObject")
        inspectedProperty(deployedId, "startingWeight", AdminConfig.showAttribute(deploymentObjectId, "startingWeight"))

        roleMappings = getSecurityRoleMappings(app)
        if len(roleMappings.keys()) > 0:
            inspectedProperty(deployedId, "roleMappings", roleMappings)

        sharedLibs = resolveSharedLibraries(app)
        if len(sharedLibs) > 0:
            inspectedProperty(deployedId, "sharedLibraries",sharedLibs)

        if  appInspectCallback:
            appInspectCallback(deployedId, app, deploymentObjectId, c)

        inspectedItem(deployedId)
