/*
 * @(#)SqlScriptOrderer.java     1 Sep 2011
 *
 * Copyright © 2010 Andrew Phillips.
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */
package com.xebialabs.deployit.plugin.was.contributor.webserver;

import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.collect.ImmutableSet.Builder;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.was.container.ManagedWebServer;
import com.xebialabs.deployit.plugin.was.container.WasContainer;
import com.xebialabs.deployit.plugin.was.contributor.CollectDeployedsOfTypesContributor;
import com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedArtifact;

import static com.google.common.base.Predicates.or;
import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.filter;
import static com.xebialabs.deployit.plugin.was.util.Predicates.instanceOf;

public abstract class AbstractReferencedWebserverContributor extends CollectDeployedsOfTypesContributor<ExtensibleDeployedArtifact> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractReferencedWebserverContributor.class);

    static final String WEBSERVERS_PROPERTY = "webServers";
    static final String WEBSERVER_NAMES_PROPERTY = "webServerNames";
    static final String WEBMODULES_PROPERTY = "webModules";

    private static final Predicate<ConfigurationItem> IS_EAR_OR_WAR_MODULE =
        or(instanceOf(Type.valueOf("was.EarModule")),
            instanceOf(Type.valueOf("was.WarModule")));

    public AbstractReferencedWebserverContributor() {
        super(Type.valueOf(ExtensibleDeployedArtifact.class));
    }

    protected Set<ManagedWebServer> getReferencedWebservers(DeploymentPlanningContext ctx) {
        Builder<ManagedWebServer> webservers = ImmutableSet.builder();

        // gather all referenced webservers
        for (ExtensibleDeployedArtifact earOrWarModule : getEarAndWarModules()) {
            Set<ManagedWebServer> referencedWebservers = earOrWarModule.getProperty(WEBSERVERS_PROPERTY);
            if (referencedWebservers != null) {
                webservers.addAll(referencedWebservers);
            }

            Set<String> webServerNames = Sets.newConcurrentHashSet();
            Set<ConfigurationItem> webModules = earOrWarModule.getProperty(WEBMODULES_PROPERTY);
            for (ConfigurationItem webModule : webModules) {
                Set<String> names = webModule.getProperty(WEBSERVER_NAMES_PROPERTY);
                webServerNames.addAll(names);
            }
            {
                Set<String> names = earOrWarModule.getProperty(WEBSERVER_NAMES_PROPERTY);
                webServerNames.addAll(names);
            }
            WasContainer container = earOrWarModule.getContainer();
            webservers.addAll(findWebServers(ctx, container, webServerNames));

        }
        return webservers.build();
    }

    private Iterable<? extends ManagedWebServer> findWebServers(DeploymentPlanningContext ctx, WasContainer container, Set<String> webServerNames) {
        Builder<ManagedWebServer> webServers = ImmutableSet.builder();
        for (String webServerName : webServerNames) {
            ManagedWebServer server = findWebServer(ctx, container, webServerName);
            if (null != server) {
                webServers.add(server);
            }
        }
        return webServers.build();
    }

    private ManagedWebServer findWebServer(DeploymentPlanningContext ctx, WasContainer container, String webServerName) {
        ManagedWebServer server = null;
        if (webServerName.startsWith("WebSphere:")) {
            // find by matching id like WebSphere:cell=vagrantCell01,node=ihsnode,server=webserver1
            server = findWebServer(ctx, container, webServerName, filterByFullName(webServerName));
        }
        else {
            server = findWebServer(ctx, container, webServerName, filterByName(webServerName));
        }
        return server;
    }

    private ManagedWebServer findWebServer(DeploymentPlanningContext ctx, WasContainer container, String webServerName, Predicate<ManagedWebServer> predicate) {
        ManagedWebServer server = null;

        Repository repository = ctx.getRepository();
        final String cellId = container.getCell().getId();

        List<ManagedWebServer> webServers = repository.search(Type.valueOf(ManagedWebServer.class));

        List<ManagedWebServer> webServersUnderSameCell = Lists.newArrayList(filter(webServers, new Predicate<ManagedWebServer>() {
            @Override
            public boolean apply(ManagedWebServer input) {
                return input.getId().startsWith(cellId + "/");
            }
        }));

        List<ManagedWebServer> foundServers = Lists.newArrayList(filter(webServersUnderSameCell, predicate));

        if (foundServers.size() == 1) {
            server = foundServers.get(0);
        }
        else if (foundServers.size() > 1) {
            logger.error("Found more than one web server with name: {} under cell {}", webServerName, cellId);
            throw new IllegalStateException(String.format("Found more than one web server with name: %s under cell %s", webServerName, cellId));
        }
        else {
            logger.error("Didn't found web server with name: {} under cell {}", webServerName, cellId);
            throw new IllegalStateException(String.format("Didn't found web server with name: %s under cell %s", webServerName, cellId));
        }

        return server;
    }

    private Predicate<ManagedWebServer> filterByName(final String webServerName) {
        return new Predicate<ManagedWebServer>() {
            @Override
            public boolean apply(ManagedWebServer input) {
                return input.getName().equals(webServerName);
            }
        };
    }

    protected Predicate<ManagedWebServer> filterByFullName(final String webServerName) {
        try {
            Pattern p = Pattern.compile("WebSphere:cell=(.*),node=(.*),server=(.*)");
            Matcher matcher = p.matcher(webServerName);
            matcher.matches();
            final String serverCellName = matcher.group(1);
            final String serverNodeName = matcher.group(2);
            final String serverName = matcher.group(3);

            return new Predicate<ManagedWebServer>() {
                @Override
                public boolean apply(ManagedWebServer input) {
                    return input.getNode().getCell().getCellName().equals(serverCellName)
                        && input.getName().equals(serverName)
                        && input.getNode().getNodeName().equals(serverNodeName);
                }
            };
        } catch (Throwable t) {
            String msg = String.format("Unable to create filter for web server name '%s'", webServerName);
            logger.error(msg, t);
            throw new IllegalStateException(msg, t);
        }
    }

    private Set<ExtensibleDeployedArtifact> getEarAndWarModules() {
        Builder<ExtensibleDeployedArtifact> earAndWarModules = ImmutableSet.builder();
        earAndWarModules.addAll(filter(concat(deployedsCreated, deployedsRemoved), IS_EAR_OR_WAR_MODULE));
        Iterable<TypedDelta> earAndWarModuleDeltas = filter(concat(deployedsModified, deployedsNoop),
            new Predicate<TypedDelta>() {
                @Override
                public boolean apply(TypedDelta input) {
                    return IS_EAR_OR_WAR_MODULE.apply(input.getDeployed());
                }
            });
        for (TypedDelta earOrWarModuleDelta : earAndWarModuleDeltas) {
            earAndWarModules.add(earOrWarModuleDelta.getPrevious());
            earAndWarModules.add(earOrWarModuleDelta.getDeployed());
        }
        return earAndWarModules.build();
    }
}