#
# utility functions
def isApplicationRunningOnServer(applicationName, serverName, nodeName) :
    return AdminControl.completeObjectName("type=Application,name=%s,process=%s,node=%s,*" % (applicationName, serverName, nodeName)) != ""

def getApplicationEditionName(deployed):
    appName = deployed.name
    if deployed.isEdition and deployed.appEdition:
        appName = "%s-edition%s" % (appName, deployed.appEdition)
    return appName

def getDeployedObject():
    appName = getApplicationEditionName(deployed)
    deployment = AdminConfig.getid('/Deployment:%s' % appName)
    return AdminConfig.showAttribute(deployment, 'deployedObject')

def getModules(deployedObject):
    return wsadminToList(AdminConfig.showAttribute(deployedObject, "modules"))

def isWebModule(id):
    return id.find('#WebModuleDeployment_') != -1

def getWebModules(deployedObject):
    return [module for module in getModules(deployedObject) if isWebModule(module)]

def moduleEquals(deployed, module):
    uri = AdminConfig.showAttribute(module, "uri").lower()
    return deployed.name.lower() == uri or (deployed.uri and deployed.uri.lower().startswith(uri))

def findModule(deployedObject, deployed):
    return validateNotEmpty([module for module in getModules(deployedObject) if moduleEquals(deployed, module)], "Unable to find module '%s'" % deployed.name)[0]

def toAdminAppArgs(properties):
    args = []
    for propName, propValue in properties.items():
        if propValue != None:
            args.append([propName, str(propValue)])
    return args

def sanitize(name):
    return ''.join([c for c in name if c.isalnum()])

def uriMapping(deployed, args=None):
    name = ".*"
    uri = ".*"
    if hasattr(deployed, "moduleDisplayName") and deployed.moduleDisplayName:
        name = deployed.moduleDisplayName
    if hasattr(deployed, "uri"):
        if deployed.uri:
            uri = deployed.uri
        elif deployed.name.endswith(".war"):
            uri = deployed.name + ",WEB-INF/web.xml"
        elif deployed.name.endswith(".jar"):
            uri = deployed.name + ",META-INF/ejb-jar.xml"
        else:
            printErrorAndExit("Please specify module name '%s' with extension (.war, .jar) or specify the URI property.", deployed.name)
    mapping = [name, uri]
    if args:
        if isinstance(args, type([])):
            mapping.extend(args)
        else:
            mapping.append(args)
    return mapping

def appendModuleMapping(moduleMappings, deployed, container, mappingCallback):
    moduleMapping = mappingCallback(deployed, container)
    if moduleMapping:
        moduleMappings.extend(moduleMapping)

def getModuleMappings(deployed, mappingCallback):
    moduleMappings = []
    container = deployed.container
    appendModuleMapping(moduleMappings, deployed, container, mappingCallback)
    if hasattr(deployed, "webModules") and deployed.webModules:
        for webModule in deployed.webModules:
            appendModuleMapping(moduleMappings, webModule, container, mappingCallback)
    if hasattr(deployed, "ejbModules") and deployed.ejbModules:
        for ejbModule in deployed.ejbModules:
            appendModuleMapping(moduleMappings, ejbModule, container, mappingCallback)
    return moduleMappings

# install argument functions
def appendMapModulesToServersArg(installArgs):
    if hasattr(deployed, 'webServers') and deployed.webServers and hasattr(deployed, 'webServerNames') and deployed.webServerNames:
        printErrorAndExit("ERROR: Both properties 'webServers' and 'webServerNames' are set, please use only 'webServerNames'.")

    targetType=deployed.container.wasTargetType

    #Check The target type,whether cluster or server
    if(targetType=='cluster'):
        defaultTargets = "WebSphere:cell=%s,%s=%s" % (deployed.container.cellName, deployed.container.wasTargetType, deployed.container.name)
    else:
        defaultTargets = "WebSphere:cell=%s,node=%s,%s=%s" % (deployed.container.cellName, deployed.container.nodeName, deployed.container.wasTargetType, deployed.container.name)

    servermappings = []
    servermappings.append(['.*', '.*', defaultTargets])

    webTargets = defaultTargets
    if hasattr(deployed, 'webServerNames') and len(deployed.webServerNames) > 0:
        webTargets += "+%s" % getWebServerNames(deployed)
    elif hasattr(deployed, 'webServers') and len(deployed.webServers) > 0:
        print "WARN: Property 'webServers' is deprecated, please use 'webServerNames' property instead."
        for ws in deployed.webServers:
            webTargets += "+%s" % toWebTarget(ws.node.cellName, ws.node.nodeName, ws.name)
    if webTargets != defaultTargets:
        servermappings.append(['.*', '.*.war,.*', webTargets])

    if hasattr(deployed, "webModules") and deployed.webModules:
        for webModule in deployed.webModules:
            if hasattr(webModule, 'webServerNames') and len(webModule.webServerNames) > 0:
                webServerNames = getWebServerNames(webModule)
                if webServerNames:
                    servermappings.append(uriMapping(webModule, "%s+%s" % (defaultTargets, webServerNames)))

    installArgs.extend(['-MapModulesToServers', servermappings])

def getWebServerNames(deployed):
    webTargets = ""
    for ws in deployed.webServerNames:
        if ws.startswith('WebSphere'):
            webTargets += "+%s" % ws
        else:
            webTargets += "+%s" % matchNameToWebServer(ws)
    return webTargets[1:]

def matchNameToWebServer(webServerName):
    print "Matching web server with name '%s' to WebSphere topology" % webServerName
    webServers = [ws for ws in wsadminToList(AdminConfig.list('WebServer')) if ws.find("/servers/%s|" % webServerName) > -1]
    numWebServers = len(webServers)
    if numWebServers == 0:
        printErrorAndExit("ERROR: Unable to find web server with name '%s'" % webServerName)
    elif numWebServers > 1:
        print "Found %s web servers matching name '%s'" % (numWebServers, webServerName)
        for ws in webServers:
            print convertWsIdToWebTarget(ws)
        printErrorAndExit("ERROR: Unable to determine target web server. Please provide the full WebSphere ID from the provided list.")
    else:
        webTarget = convertWsIdToWebTarget(webServers[0])
        print "Matched web server name '%s' to target server: '%s'" % (webServerName, webTarget)
        return webTarget

def convertWsIdToWebTarget(wsId):
    match = re.match(r".*\(cells\/(.*)\/nodes\/(.*)\/servers\/(.*)\|.*\)", wsId)
    return toWebTarget(*match.groups())

def convertWebTargetToContainmentPath(webTarget):
    return webTarget.replace("WebSphere:cell=", "/Cell:").replace(",node=", "/Node:").replace(",server=", "/Server:")

def toWebTarget(cellName, nodeName, serverName):
    return "WebSphere:cell=%s,node=%s,server=%s" % (cellName, nodeName, serverName)

def appendMapWebModToVHArg(installArgs):
    vhosts = getModuleMappings(deployed, getVirtualHostMappings)
    if vhosts:
        installArgs.extend(['-MapWebModToVH', vhosts])

def getVirtualHostMappings(deployed, container):
    if not hasattr(deployed, 'virtualHostName') and not hasattr(deployed, 'virtualHost'):
        return None
    vhostName = "default_host"
    if hasattr(deployed, 'virtualHost') and deployed.virtualHost and hasattr(deployed, 'virtualHostName') and deployed.virtualHostName:
        printErrorAndExit("ERROR: Both properties 'virtualHost' and 'virtualHostName' are set, please use only 'virtualHostName'.")
    if hasattr(deployed, 'virtualHostName') and deployed.virtualHostName:
        vhostName = deployed.virtualHostName
    elif hasattr(deployed, 'virtualHost') and deployed.virtualHost:
        vhostName = deployed.virtualHost.name
        print "WARN: Property 'virtualHost' is deprecated, please use 'virtualHostName' property instead."
    if vhostName:
        return [uriMapping(deployed, vhostName)]

def appendMapSharedLibForModArg(installArgs):
    sharedLibs = getModuleMappings(deployed, getSharedLibMappings)
    if sharedLibs:
        installArgs.extend(['-MapSharedLibForMod', sharedLibs])

def getSharedLibMappings(deployed, container):
    sharedLibNames = []
    if hasattr(deployed, "sharedLibraries") and deployed.sharedLibraries:
        for l in deployed.sharedLibraries:
            sharedLibNames.append(l.name)
    if hasattr(deployed, "sharedLibraryNames") and deployed.sharedLibraryNames:
        for l in deployed.sharedLibraryNames:
            if not l in sharedLibNames:
                sharedLibNames.append(l)
            else:
                print "WARN: duplicate shared library name '%s' while deploying '%s'" % (l, deployed.name)
    if sharedLibNames:
        return [uriMapping(deployed, "+".join(sharedLibNames))]

def appendMapRolesToUsersArg(installArgs):
    roleMappings = []
    # group mapping
    if hasattr(deployed, 'roleMappings'):
        for role,group in deployed.roleMappings.items():
            # special case - allow specification of 'All Role' without requiring spaces
            if role == 'All':
                role = 'All Role'

            if group == 'Everyone':
                roleMappings.append([role, 'Yes', 'No', '', ''])
            elif group == 'AllAuthenticated':
                roleMappings.append([role, 'No', 'Yes', '', ''])
            elif group == 'AllAuthenticatedInTrustedRealms':
                roleMappings.append([role, 'No', 'No', '', '', 'Yes', '', ''])
            else:
                roleMappings.append([role, 'No', 'No', '', group])
    # user mapping
    if hasattr(deployed, 'roleUserMappings'):
        for role, user in deployed.roleUserMappings.items():
            mappings = [roleMapping for roleMapping in roleMappings if roleMapping[0] == role]
            if mappings:
                mappings[0][3] = user
            else:
                roleMappings.append([role, 'No', 'No', user, ''])

    if len(roleMappings) > 0:
        installArgs.extend(['-MapRolesToUsers', roleMappings])

def appendMapRunAsRolesToUsersArg(installArgs):
    if hasattr(deployed, 'runAsRoleMappings'):
        runAsRoleMappings = []
        for roleUser in deployed.runAsRoleMappings.keys():
            if roleUser.endswith("_user"):
                role = roleUser[:roleUser.rindex("_user")]
                user = deployed.runAsRoleMappings[roleUser]
                pwd = deployed.runAsRoleMappings["%s_password" % role]
                if not pwd:
                    printErrorAndExit("ERROR: %s defined in runAsRoleMappings property must have a matching %s_password entry" % (roleUser, role))
                runAsRoleMappings.append([role, user, pwd])

        if len(runAsRoleMappings) > 0:
            installArgs.extend(['-MapRunAsRolesToUsers', runAsRoleMappings])

def appendPrecompileJspsArg(installArgs):
    jspCompilationOptions = getModuleMappings(deployed, getJspCompilationMappings)
    if jspCompilationOptions:
        installArgs.extend(['-preCompileJSPs','-JSPCompileOptions', jspCompilationOptions])

def getJspCompilationMappings(deployed, container):
    if hasattr(deployed, 'preCompileJsps') and deployed.preCompileJsps:
        return [uriMapping(deployed, [deployed.jspClassPath, deployed.useFullPackageNames, deployed.jdkSourceLevel, deployed.disableJspRuntimeCompilation])]

def appendJspReloadForWebModArg(installArgs):
    jspReloadMappings = getModuleMappings(deployed, getJspReloadMappings)
    if jspReloadMappings:
        installArgs.extend(['-JSPReloadForWebMod', jspReloadMappings])

def getJspReloadMappings(deployed, container):
    if hasattr(deployed, 'enableJspClassReloading'):
        if deployed.enableJspClassReloading and noneToEmpty(deployed.jspReloadInterval) != "":
            mapping = uriMapping(deployed, ['Yes', str(deployed.jspReloadInterval)])
        else:
            mapping = uriMapping(deployed, ['No', ''])
        if getVersion(container) == "WAS_61" or getVersion(container) == "WAS_70":
            mapping[1] = mapping[1].replace("/web.xml", "/ibm-web-ext.xmi")
        return [mapping]

def getRefMappings(deployed, container, propertyName, callbackFunc=None):
    if hasattr(deployed, propertyName) and getattr(deployed, propertyName):
        refs = []
        moduleName, uri = uriMapping(deployed)
        for ref in getattr(deployed, propertyName):
            refMapping = [moduleName, ref.ejbName, uri, ref.resourceRefName, ref.resourceType, ref.resourceJndiName]
            if callbackFunc:
                callbackFunc(deployed, container, ref, refMapping)
            refs.append(refMapping)
        return refs

def appendMapEJBRefToEJB(installArgs):
    ejbRefs = getModuleMappings(deployed, getEJBRefMappings)
    if ejbRefs:
        installArgs.extend(['-MapEJBRefToEJB', ejbRefs])

def getEJBRefMappings(deployed, container):
    return getRefMappings(deployed, container, "ejbRefs")

def appendMapResEnvRefToRes(installArgs):
    resEnvRefs = getModuleMappings(deployed, getResEnvRefMappings)
    if resEnvRefs:
        installArgs.extend(['-MapResEnvRefToRes', resEnvRefs])

def getResEnvRefMappings(deployed, container):
    return getRefMappings(deployed, container, "resourceEnvRefs")

def appendMapResRefToEJB(installArgs):
    resRefs = getModuleMappings(deployed, getResRefMappings)
    if resRefs:
        installArgs.extend(['-MapResRefToEJB', resRefs])

def getResRefMappings(deployed, container):
    return getRefMappings(deployed, container, "resourceRefs", getResRefMappingsCallback)

def getResRefMappingsCallback(deployed, container, ref, mapping):
    mapping.extend([ref.loginConfigurationName, ref.properties])
    if getVersion(container) != 'WAS_61':
        mapping.append('')

def appendEarDestinationArg(installArgs):
    # optionally, specify a custom directory to store the installed module (if set on the was.Module)
    if hasattr(deployed, 'moduleDestination') and deployed.moduleDestination:
        installArgs.extend(['-installed.ear.destination', deployed.moduleDestination])

    # optionally, specify a custom directory to store the installed module (if set on the was.Container)
    elif hasattr(deployed.container, 'moduleDestination') and deployed.container.moduleDestination:
        installArgs.extend(['-installed.ear.destination', deployed.container.moduleDestination])

def appendDeployEjbArg(installArgs):
    deployEjbArgs = []
    if hasattr(deployed, 'ejbDbType') and deployed.ejbDbType:
        deployEjbArgs.extend(['-deployejb.dbtype ', deployed.ejbDbType])
    if hasattr(deployed, 'ejbDbSchema') and deployed.ejbDbSchema:
        deployEjbArgs.extend(['-deployejb.dbschema', deployed.ejbDbSchema])
    if hasattr(deployed, 'ejbClasspath') and deployed.ejbClasspath:
        deployEjbArgs.extend(['-deployejb.classpath', deployed.ejbClasspath])

    if len(deployEjbArgs) > 0:
        deployEjbArgs.append('-deployejb')
        installArgs.extend(deployEjbArgs)

def appendContextRootArg(installArgs):
    # if this is not a WAR deployment, the contextroot property will not be present on the deployed
    if hasattr(deployed, 'contextRoot'):
        installArgs.extend(['-contextroot', deployed.contextRoot])

    # if embedded modules are specified, update their context root settings
    ctxRootForWebMods = getModuleMappings(deployed, getCtxRootForWebModMappings)
    if ctxRootForWebMods:
        installArgs.extend(['-CtxRootForWebMod', ctxRootForWebMods])

def getCtxRootForWebModMappings(deployed, container):
    if hasattr(deployed, "uri") and hasattr(deployed, 'contextRoot') and deployed.contextRoot:
        return [uriMapping(deployed, deployed.contextRoot)]

def appendEditionedApplicationArgs(installArgs):
    if hasattr(deployed, 'isEdition') and deployed.isEdition:
        if hasattr(deployed, 'appEdition') and deployed.appEdition:
            installArgs.extend(['-edition', deployed.appEdition])
        if hasattr(deployed, 'editionDescription') and deployed.editionDescription:
            installArgs.extend(['-edition.desc', deployed.editionDescription])

def appendAdditionalInstallFlags(installArgs):
    for f in deployed.additionalInstallFlags:
        installArgs.append(f)

def prepareInstallArgs():
    installArgs = ['-appname', deployed.name]
    appendContextRootArg(installArgs)
    appendMapModulesToServersArg(installArgs)
    appendMapWebModToVHArg(installArgs)
    appendMapSharedLibForModArg(installArgs)
    appendMapRolesToUsersArg(installArgs)
    appendMapRunAsRolesToUsersArg(installArgs)
    appendPrecompileJspsArg(installArgs)
    appendJspReloadForWebModArg(installArgs)
    appendMapResRefToEJB(installArgs)
    appendMapEJBRefToEJB(installArgs)
    appendMapResEnvRefToRes(installArgs)
    appendEarDestinationArg(installArgs)
    appendEditionedApplicationArgs(installArgs)
    appendDeployEjbArg(installArgs)
    appendAdditionalInstallFlags(installArgs)
    return installArgs

# update functions
def updateStartingWeight(deployedObject):
    # if this is not an EAR deployment, the startingWeight property will not be present on the deployed
    if hasattr(deployed, 'startingWeight'):
        setStartingWeight(deployedObject, deployed.startingWeight)

    if hasattr(deployed, "webModules") and deployed.webModules:
        for webModule in deployed.webModules:
            setStartingWeight(findModule(deployedObject, webModule), webModule.startingWeight)
    if hasattr(deployed, "ejbModules") and deployed.ejbModules:
        for ejbModule in deployed.ejbModules:
            setStartingWeight(findModule(deployedObject, ejbModule), ejbModule.startingWeight)

def setStartingWeight(module, startingWeight):
    if startingWeight == None: return
    AdminConfig.modify(module, [['startingWeight', str(startingWeight)]])

def updateClassloader(deployedObject):
    if hasattr(deployed, 'classloaderMode') and deployed.classloaderMode:
        classLoaderAttr = AdminConfig.showAttribute(deployedObject, 'classloader')
        print "Setting application classloader to %s" % (deployed.classloaderMode)
        AdminConfig.modify(classLoaderAttr, [['mode', deployed.classloaderMode]])

    if hasattr(deployed, 'warClassloaderMode') and deployed.warClassloaderMode:
        for module in getWebModules(deployedObject):
            setModuleClassloader(module, deployed.warClassloaderMode)

    if hasattr(deployed, "webModules") and deployed.webModules:
        for webModule in deployed.webModules:
            if webModule.classloaderMode:
                setModuleClassloader(findModule(deployedObject, webModule), webModule.classloaderMode)

def setModuleClassloader(module, classloaderMode):
    if not classloaderMode: return
    warName = AdminConfig.showAttribute(module, 'uri')
    print "Setting WAR classloader for '%s' to %s" % (warName, classloaderMode)
    AdminConfig.modify(module, [['classloaderMode', classloaderMode]])

def updateSessionManager(deployedObject):
    if hasattr(deployed, 'sessionManagers') and deployed.sessionManagers:
        for sessionManager in deployed.sessionManagers:
            args = toSessionManagerArgs(sessionManager)
            print "Setting session management settings for application to %s" % args
            configId = AdminConfig.create('ApplicationConfig', deployedObject, args)
            for target in wsadminToList(AdminConfig.showAttribute(deployedObject, 'targetMappings')):
                if target.find('DeploymentTargetMapping') != -1:
                    AdminConfig.modify(target,[['config', configId]])

    if hasattr(deployed, "webModules") and deployed.webModules:
        for webModule in deployed.webModules:
            if hasattr(webModule, 'sessionManagers') and webModule.sessionManagers:
                for sessionManager in webModule.sessionManagers:
                    args = toSessionManagerArgs(sessionManager)
                    module = findModule(deployedObject, webModule)
                    moduleUri = AdminConfig.showAttribute(module, "uri")
                    print "Setting session management settings for module '%s' to %s" % (moduleUri, args)
                    configId = AdminConfig.create('WebModuleConfig', module, args)
                    for target in wsadminToList(AdminConfig.showAttribute(module, 'targetMappings')):
                        if target.find('DeploymentTargetMapping') != -1:
                            AdminConfig.modify(target,[['config', configId]])

def toSessionManagerArgs(sessionManager):
    sessionMgmtArgs = toAdminConfigArgs(excludeUnsupportedProperties('SessionManager', sessionManager.getExposedProperties(True)))
    sessionMgmtArgs.append(['enable', 'true'])

    cookieArgs = getNestedArgs(sessionManager, "Cookie")
    dbPersistenceArgs = getNestedArgs(sessionManager, "SessionDatabasePersistence")
    drsPersistenceArgs = getNestedArgs(sessionManager,"DRSSettings")
    tuningParamsArgs = getNestedArgs(sessionManager, "TuningParams")
    invalidationArgs = getNestedArgs(sessionManager, "InvalidationSchedule")

    if cookieArgs:
        sessionMgmtArgs.append(['defaultCookieSettings', cookieArgs])
    if dbPersistenceArgs:
        sessionMgmtArgs.append(['sessionDatabasePersistence', dbPersistenceArgs])
    if drsPersistenceArgs:
        sessionMgmtArgs.append(['sessionDRSPersistence', drsPersistenceArgs])
    if tuningParamsArgs:
        if invalidationArgs:
            tuningParamsArgs.append(['invalidationSchedule', invalidationArgs])
        sessionMgmtArgs.append(['tuningParams', tuningParamsArgs])

    args = [['sessionManagement', sessionMgmtArgs]]
    return args

def getNestedArgs(parentObject, nestedProperty):
    if hasattr(parentObject, nestedProperty):
        return toAdminConfigArgs(excludeUnsupportedProperties(nestedProperty, getattr(parentObject, nestedProperty).getExposedProperties(True)))

def updateJsfImplementation(deployedObject):
    if hasattr(deployed, 'jsfImplementation') and deployed.jsfImplementation:
        if getVersion(deployed.container) == 'WAS_61':
            print "WARN: Ignoring property 'jsfImplementation' which is not supported on WAS 6.1"
        else:
            print "Setting JSF implementation to '%s'" % (deployed.jsfImplementation)
            AdminTask.modifyJSFImplementation(getApplicationEditionName(deployed), "[-implName '%s']" % (deployed.jsfImplementation))

def updateOtherProperties(deployedObject):
    args = toAdminConfigArgs(excludeUnsupportedProperties("ApplicationDeployment", deployed.getExposedProperties(True), False))
    print "Setting application properties to %s" %  args
    AdminConfig.modify(deployedObject, args)

# deploy
def deployApplication():
    if deployed.file is None:
        printErrorAndExit("ERROR: No file found to deploy, cannot proceed")
    # install app
    installArgs = prepareInstallArgs()
    print "Deploying application", deployed.name, "with args: %s" % (installArgs)
    AdminApp.install(deployed.file, installArgs)

    # update app
    deployedObject = getDeployedObject()
    updateStartingWeight(deployedObject)
    updateClassloader(deployedObject)
    updateSessionManager(deployedObject)
    updateJsfImplementation(deployedObject)
    updateOtherProperties(deployedObject)
