/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.step;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.hostsession.HostFile;
import com.xebialabs.deployit.hostsession.HostSession;
import com.xebialabs.deployit.plugin.wls.ci.WlsDomain;

/**
 * Detects whether the WLS installation has a broken wlst.sh script. A broken wlst.sh script is one that cannot handle spaces in its arguments because it passes
 * them on using $* instead of "$@".
 */
@SuppressWarnings("serial")
public class DetectBrokenWlstStep extends WlsStepBase {

	public DetectBrokenWlstStep(WlsDomain domain) {
		super(domain);
		setDescription("Detect broken wlst.sh script");
	}

	public boolean execute(StepExecutionContext ctx) {
		if (domain.getActiveHost().getOperatingSystemFamily() != OperatingSystemFamily.UNIX) {
			ctx.logOutput(domain + " does not run on a Unix box. Disabling workaround.");
			domain.setEnableWlstShWorkaround(false);
			return true;
		}

		HostSession s = domain.connectToAdminHost();
		try {
			try {
				String wlstPath = domain.getWlstPath();
				ctx.logOutput("Reading " + wlstPath + " from " + domain + " to determine whether it is broken.");
				HostFile wlstSh = s.getFile(wlstPath);
				BufferedReader wlstReader = new BufferedReader(new InputStreamReader(wlstSh.get()));
				try {
					String wlstLine;
					while ((wlstLine = wlstReader.readLine()) != null) {
						if (wlstLine.contains("${JAVA_HOME}/bin/java") && wlstLine.contains("weblogic.WLST")) {
							boolean containsIncorrectExpansion = wlstLine.contains("$*");
							if (containsIncorrectExpansion) {
								ctx.logOutput("Found broken wlst.sh. Enabling workaround.");
							} else {
								ctx.logOutput("Found correct wlst.sh. Disabling workaround");
							}
							domain.setEnableWlstShWorkaround(containsIncorrectExpansion);
							return true;
						}
					}
				} finally {
					wlstReader.close();
				}
				ctx.logError("Could not determine whether wlst.sh was broken. Assuming it is correct and disabling workaround.");
			} catch (IOException exc) {
				ctx.logError("Could not determine whether wlst.sh was broken. Assuming it is correct and disabling workaround.", exc);
			}
			domain.setEnableWlstShWorkaround(false);
			return true;
		} finally {
			s.close();
		}
	}

}
