/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.step;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.artifact.NamedDeployableArtifact;
import com.xebialabs.deployit.ci.artifact.mapping.PlaceholderFormat;
import com.xebialabs.deployit.hostsession.*;
import com.xebialabs.deployit.plugin.wls.ci.WlsTarget;

import java.util.Collections;
import java.util.Map;

/**
 * Deploys (WLS8) applications on a {@link com.xebialabs.deployit.plugin.wls.ci.WlsCluster cluster} or {@link com.xebialabs.deployit.plugin.wls.ci.WlsServer managed server}.
 */

public class Wls8DeployArtifactStep extends Wls8WeblogicDeployerStepBase implements Step {

	private String extension;
	private String targetName;
	private String applicationName;
	private String applicationLocation;
	private Map<String, String> properties;
	private PlaceholderFormat placeholderFormat;


	public Wls8DeployArtifactStep(WlsTarget target, NamedDeployableArtifact artifact, Map<String, String> properties, PlaceholderFormat placeholderFormat) {
		super(target.getDomain());
		this.targetName = target.getName();
		this.applicationName = artifact.getName();
		this.extension = findOutExtension(artifact);
		this.applicationLocation = artifact.getLocation();
		this.properties = properties;
		this.placeholderFormat = placeholderFormat;


		if (properties == null || properties.isEmpty())
			setDescription("Deploy " + artifact + " on " + target);
		else
			setDescription("Deploy " + artifact + " on " + target + " while replacing properties in the archive.");
	}

	public Wls8DeployArtifactStep(WlsTarget target, NamedDeployableArtifact artifact) {
		this(target, artifact, Collections.<String, String>emptyMap(), PlaceholderFormat.SPRING);
	}

	public boolean execute(StepExecutionContext ctx) {
		HostSession rhs = domain.connectToAdminHost();
		try {
			HostFile uploadedApplication = uploadApplication(rhs);
			String uploadedApplicationPath = uploadedApplication.getPath().replace('\\', '/');

			//java weblogic.Deployer -username XXX -password YYY -adminurl t3://host:port -name applicationName -targets targetName -upload -stage -source applicationLocation -deploy
			Map<String, String> velocityContext = Maps.newHashMap();
			velocityContext.put("username", domain.getUsername());
			velocityContext.put("password", domain.getPassword());
			velocityContext.put("adminurl", domain.getConnectUrl());
			velocityContext.put("wlHome", domain.getWlHome());
			velocityContext.put("uploadedApplication", uploadedApplicationPath);
			velocityContext.put("target", targetName);
			velocityContext.put("name", applicationName);
			velocityContext.put("deployerAction", "-upload -stage -deploy " + uploadedApplicationPath);

			int res = executeWeblogicDeployerScript(ctx, rhs, velocityContext);

			return res == 0;
		} finally {
			rhs.close();
		}
	}

	private HostFileInputStreamTransformer getTransformer() {
		if (properties.isEmpty())
			return null;
		return new LenientTemplateResolvingArchiveHostFileInputTransformer(properties, placeholderFormat);
	}


	private HostFile uploadApplication(HostSession rhs) {
		HostSession lhs = Host.getLocalHost().getHostSession();
		try {
			HostFile applicationFileToUpload = lhs.getFile(applicationLocation);
			HostFile uploadedApplicationFile = rhs.getTempFile(applicationName, extension);
			HostFileUtils.copy(applicationFileToUpload, uploadedApplicationFile, getTransformer());
			return uploadedApplicationFile;
		} finally {
			lhs.close();
		}
	}


}
