package com.xebialabs.deployit.plugin.wls.step;

import static com.google.common.collect.Lists.newArrayList;
import static org.apache.commons.lang.StringUtils.join;
import static org.apache.commons.lang.StringUtils.split;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.StepExecutionContext;
import com.xebialabs.deployit.plugin.wls.ci.WlsServer;
import com.xebialabs.deployit.plugin.wls.ci.WlsServerConfigurationModification;

@SuppressWarnings("serial")
public class ApplyWlsServerConfigurationModificationStep extends WlsStepBase {

	private WlsServer server;

	private WlsServerConfigurationModification modificationToApply;

	private WlsServerConfigurationModification modificationToUnapply;

	public ApplyWlsServerConfigurationModificationStep(WlsServer server, WlsServerConfigurationModification modificationToApply,
	        WlsServerConfigurationModification modificationToUnapply) {
		super(server.getDomain());
		this.server = server;
		this.modificationToApply = modificationToApply;
		this.modificationToUnapply = modificationToUnapply;
		StringBuilder desc = new StringBuilder();
		if (modificationToApply != null) {
			if (modificationToUnapply != null) {
				desc.append("Remove \"" + modificationToUnapply.getClasspathAddition() + "\" and add \"" + modificationToApply.getClasspathAddition()
				        + "\" to classpath of " + server);
			} else {
				desc.append("Add \"" + modificationToApply.getClasspathAddition() + "\" to classpath of " + server);
			}
		} else if (modificationToUnapply != null) {
			desc.append("Remove \"" + modificationToUnapply.getClasspathAddition() + "\" fron classpath of " + server);
		} else {
			throw new IllegalArgumentException("Either modificationToApply or modificationToUnapply should be non-null");
		}

		setDescription(desc.toString());
	}

	@Override
	public boolean execute(StepExecutionContext ctx) {
		boolean couldRetrieveServerConfiguration = new RetrieveWlsServerStep(server).execute(ctx);
		if (!couldRetrieveServerConfiguration) {
			ctx.logError("Cannot retrieve configuration of server " + server);
			return false;
		}
		ctx.logOutput("********");

		String originalClasspath = server.getClasspath();
		if(originalClasspath == null){
			originalClasspath = StringUtils.EMPTY;
		}
		logger.debug("Original classpath: {}", originalClasspath);

		String pathSeparator = server.getDomain().getActiveHost().getOperatingSystemFamily().getPathSeparator();
		List<String> classpathComponents = newArrayList(split(originalClasspath, pathSeparator));
		logger.debug("Original classpath components: {}", classpathComponents);

		boolean classpathWasModified = false;

		if (modificationToUnapply != null) {
			List<String> classpathComponentsToRemove = newArrayList(split(modificationToUnapply.getClasspathAddition(), pathSeparator));
			logger.debug("Classpath components to remove: {}", classpathComponentsToRemove);

			for (String c : classpathComponentsToRemove) {
				if (classpathComponents.contains(c)) {
					classpathComponents.remove(c);
					classpathWasModified = true;
				}
			}
		}

		if (modificationToApply != null) {
			List<String> classpathComponentsToAdd = newArrayList(split(modificationToApply.getClasspathAddition(), pathSeparator));
			logger.debug("Classpath components to add: {}", classpathComponentsToAdd);

			for (String c : classpathComponentsToAdd) {
				if (!classpathComponents.contains(c)) {
					classpathComponents.add(c);
					classpathWasModified = true;
				}
			}
		}

		logger.debug("Modified classpath components: {}", classpathComponents);

		if (!classpathWasModified) {
			ctx.logOutput("Classpath does not have to be updated. Done.");
			return true;
		}
		String modifiedClasspath = join(classpathComponents, pathSeparator);
		logger.debug("Modified classpath: {}", modifiedClasspath);

		ctx.logOutput("Modified server classpath: " + modifiedClasspath);
		ctx.logOutput("********");

		server.setClasspath(modifiedClasspath);
		if (!new ModifyWlsServerStep(server).execute(ctx))
			return false;
		ctx.logOutput("********");

		if (!new ActivatePendingChangesStep(server.getDomain()).execute(ctx))
			return false;
		ctx.logOutput("********");

		if (!new StopWlsServerStep(server).execute(ctx))
			return false;
		ctx.logOutput("********");

		if (!new StartWlsServerStep(server).execute(ctx))
			return false;
		ctx.logOutput("********");

		return true;
	}

	public WlsServerConfigurationModification getModificationToApply() {
		return modificationToApply;
	}

	public WlsServerConfigurationModification getModificationToUnapply() {
		return modificationToUnapply;
	}

	private static Logger logger = LoggerFactory.getLogger(ApplyWlsServerConfigurationModificationStep.class);

}
