package com.xebialabs.deployit.plugin.wls.validator;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.plugin.api.deployment.planning.PrePlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.wls.container.Cluster;
import com.xebialabs.deployit.plugin.wls.container.Domain;
import com.xebialabs.deployit.plugin.wls.container.JmsServer;
import com.xebialabs.deployit.plugin.wls.container.JmsTarget;
import com.xebialabs.deployit.plugin.wls.deployed.AbstractQueue;
import com.xebialabs.deployit.plugin.wls.deployed.AbstractUniformDistributedQueue;
import com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedArtifact;
import com.xebialabs.deployit.plugin.wls.deployed.JmsResource;
import com.xebialabs.deployit.plugin.wls.resource.JmsResourceSpec;

import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Multimaps.index;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.CREATE;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;
import static java.util.EnumSet.of;

public class PlanLevelValidator {

    public static final Predicate<Delta> CREATE_OR_MODIFY_QUEUE_FILTER = new Predicate<Delta>() {
        @Override
        public boolean apply(Delta input) {
            return of(CREATE, MODIFY).contains(input.getOperation()) && AbstractQueue.class.isAssignableFrom(input.getDeployed().getClass());
        }
    };

    @PrePlanProcessor
    public Step validatePlan(DeltaSpecification spec) {
        validateArtifactDeploymentOptions(spec);
        validateUniqueJNDIName(spec);
        validateUniformDistributedQueues(spec);
        validateQueueIsTargetedToSingleJmsServerWithinDomain(spec);
        return null;
    }

    private static void validateArtifactDeploymentOptions(DeltaSpecification spec) {
        Multimap<Deployable, ExtensibleDeployedArtifact<DeployableArtifact>> deployeds = getAllDeployedArtifacts(spec);

        for (Deployable eachArtifact : deployeds.keySet()) {
            Collection<ExtensibleDeployedArtifact<DeployableArtifact>> artifactDeployeds = deployeds.get(eachArtifact);
            validateDeploymentOptionsAreSame(artifactDeployeds);
        }

    }

    @SuppressWarnings("unchecked")
    private static Multimap<Deployable, ExtensibleDeployedArtifact<DeployableArtifact>> getAllDeployedArtifacts(DeltaSpecification spec) {
        Multimap<Deployable, ExtensibleDeployedArtifact<DeployableArtifact>> deployeds = HashMultimap.create();
        for (Delta each : spec.getDeltas()) {
            if ((each.getOperation() == CREATE || each.getOperation() == MODIFY)
                    && ExtensibleDeployedArtifact.class.isAssignableFrom(each.getDeployed().getClass())) {
                ExtensibleDeployedArtifact<DeployableArtifact> artifact = (ExtensibleDeployedArtifact<DeployableArtifact>) each.getDeployed();
                deployeds.put(artifact.getDeployable(), artifact);
            }
        }
        return deployeds;
    }

    private static void validateDeploymentOptionsAreSame(Collection<ExtensibleDeployedArtifact<DeployableArtifact>> artifactDeployeds) {
        Iterator<ExtensibleDeployedArtifact<DeployableArtifact>> iterator = artifactDeployeds.iterator();
        if (iterator.hasNext()) {
            ExtensibleDeployedArtifact<DeployableArtifact> reference = iterator.next();

            while (iterator.hasNext()) {
                ExtensibleDeployedArtifact<DeployableArtifact> next = iterator.next();
                validateSameValues(reference, next);
            }
        }
    }

    private static void validateSameValues(ExtensibleDeployedArtifact<DeployableArtifact> reference, ExtensibleDeployedArtifact<DeployableArtifact> deployed) {
        if (reference.getContainer().getDomain().equals(deployed.getContainer().getDomain())) {
            if (reference.isVersioned() != deployed.isVersioned()) {
                throw new RuntimeException("Can't have different values for 'versioned' property for same artifact " + reference.getDeployable().getName());
            }
            if (reference.getStageMode() != deployed.getStageMode()) {
                throw new RuntimeException("Can't have different values for 'stageMode' property for same artifact " + reference.getDeployable().getName());
            }
            if (reference.getStagingDirectory() != null && !reference.getStagingDirectory().equals(deployed.getStagingDirectory())) {
                throw new RuntimeException("Can't have different values for 'stagingDirectory' property for same artifact "
                        + reference.getDeployable().getName());
            }
            if (reference.getRedeploymentStrategy() != deployed.getRedeploymentStrategy()) {
                throw new RuntimeException("Can't have different values for the deployment strategy for the same artifact "
                        + reference.getDeployable().getName());
            }
        }
    }

    @SuppressWarnings({ "unchecked" })
    private static void validateUniqueJNDIName(DeltaSpecification spec) {
        Map<String, String> jndiNames = new HashMap<String, String>();
        for (Delta each : spec.getDeltas()) {
            if ((each.getOperation() == CREATE || each.getOperation() == MODIFY) && JmsResource.class.isAssignableFrom(each.getDeployed().getClass())) {
                JmsResource<JmsResourceSpec, JmsTarget> jmsResource = (JmsResource<JmsResourceSpec, JmsTarget>) each.getDeployed();
                final String jndiName = jmsResource.getJndiName();
                //final String name = jmsResource.getDeployable().getName();
                final String name = jmsResource.getName();
                if (jndiNames.get(jndiName) != null && !jndiNames.get(jndiName).equals(name)) {
                    throw new RuntimeException(String.format("Jndi name %s is in use by multiple resource sources", jndiName));
                } else {
                    jndiNames.put(jndiName, name);
                }
            }
        }
    }

    private static void validateUniformDistributedQueues(DeltaSpecification spec) {
        // A UDD must be targeted to servers within a single cluster or a single
        // stand-alone server
        Map<String, Cluster> uddQueueContainers = new HashMap<String, Cluster>();
        for (Delta each : spec.getDeltas()) {
            if ((each.getOperation() == CREATE || each.getOperation() == MODIFY)
                    && AbstractUniformDistributedQueue.class.isAssignableFrom(each.getDeployed().getClass())) {
                AbstractUniformDistributedQueue jmsResource = (AbstractUniformDistributedQueue) each.getDeployed();
                Cluster cluster = getContainerCluster(jmsResource);
                // the cluster to which the container belongs or null if it points to a stand-alone server
                final String jmsResourceName = jmsResource.getName();
                if (uddQueueContainers.containsKey(jmsResourceName) && !uddQueueContainers.get(jmsResourceName).equals(cluster)) {
                    throw new RuntimeException(String.format("The UDD %s must be targeted to servers within a single cluster or a single stand-alone server",
                            jmsResourceName));
                } else {
                    uddQueueContainers.put(jmsResourceName, cluster);
                }
            }
        }
    }

    private static void validateQueueIsTargetedToSingleJmsServerWithinDomain(DeltaSpecification spec) {
        // First find all the queues that are either CREATE or MODIFY.
        FluentIterable<AbstractQueue> queues = from(spec.getDeltas()).filter(CREATE_OR_MODIFY_QUEUE_FILTER).transform(new Function<Delta, AbstractQueue>() {
            @Override
            public AbstractQueue apply(Delta input) {
                return (AbstractQueue) input.getDeployed();
            }
        });

        // Index the queues by name.
        ImmutableListMultimap<String, AbstractQueue> queueByName = index(queues, new Function<AbstractQueue, String>() {
            @Override
            public String apply(AbstractQueue input) {
                return input.getName();
            }
        });

        // For each of the deployed queues, check whether it is deployed more than once to the same domain.
        // This uses set-theory to find duplicate domains (ie. the set of domains is smaller than the total collection of domains)
        for (String queueName : queueByName.keySet()) {
            if (checkDuplicates(Lists.transform(queueByName.get(queueName), new Function<AbstractQueue, Domain>() {
                @Override
                public Domain apply(AbstractQueue input) {
                    return input.getContainer().getDomain();
                }
            }))) {
                throw new RuntimeException((String.format("Queue %s must be targeted to only a single JmsServer in the same Domain.", queueName)));
            }
        }
    }

    private static <T> boolean checkDuplicates(List<T> list) {
        return newHashSet(list).size() < list.size();
    }

    private static Cluster getContainerCluster(AbstractUniformDistributedQueue jmsResource) {
        JmsTarget container = jmsResource.getContainer();
        Cluster cluster = null;
        if (Cluster.class.isAssignableFrom(container.getClass())) {
            cluster = (Cluster) container;
        } else if (JmsServer.class.isAssignableFrom(container.getClass())) {
            cluster = ((JmsServer) container).getServer().getClusterIfAny();
        }
        return cluster;
    }

}
