/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.container;

import java.util.*;
import com.google.common.base.Function;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.python.ControlTaskDelegate;
import com.xebialabs.deployit.plugin.python.PythonInspectionStep;
import com.xebialabs.deployit.plugin.python.PythonManagingContainer;

import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Lists.newArrayList;

/**
 * A cluster in a {@link Domain WLS domain}.
 *
 * @see Server
 */
@SuppressWarnings("serial")
@Metadata(description = "WebLogic Cluster which defines groups of WebLogic servers that work together to increase scalability and reliability")
public class Cluster extends BaseContainer implements WlsContainer, JmsTarget {
    protected final Map<String, Object> context;

    private static final String START_CONTROL_TASK = "start";
    private static final String STOP_CONTROL_TASK = "stop";

    @Property(required = true, asContainment = true, description = "The domain to which the WebLogic Cluster belongs. 'asContainment'=true, means a Cluster is 'contained' under a Domain", category = "Topology")
    private Domain domain;

    @Property(description = "Servers in the WebLogic Cluster", required = false)
    private Set<Server> servers = new HashSet<Server>();

    @Property(required = true, defaultValue = "80", hidden = true)
    private int startOrder;

    @Property(required = true, defaultValue = "20", hidden = true)
    private int stopOrder;

    @Property(required = true, hidden = true, defaultValue = "1" ,description = "The time to wait in seconds for a container start action.")
    private int startWaitTime;

    public Cluster() {
        context = new HashMap<>();
        context.put("container", (Object) this);
    }

    public String getObjectName() {
        return "com.bea:Name=" + getName() + ",Type=Cluster";
    }

    @Inspect
    public void inspect(InspectionContext ctx) {
        Map<String, Object> pythonVars = Maps.newHashMap();
        pythonVars.put("container", this);
        ctx.addStep(new PythonInspectionStep(this, this.getManagingContainer(), "wls/container/inspect-cluster.py", pythonVars, "Inspect " + this));
    }

    @Override
    public Set<Host> getHosts() {
        Set<Host> hosts = Sets.newHashSet();
        for (Server server : servers)
            hosts.addAll(server.getHosts());
        return hosts;
    }

    public Domain getDomain() {
        return domain;
    }

    public void setDomain(Domain domain) {
        this.domain = domain;
    }

    public Set<Server> getServers() {
        return Collections.unmodifiableSet(servers);
    }

    public void setServers(Set<Server> servers) {
        this.servers = servers;
    }

    public void addServer(Server server) {
        this.servers.add(server);
    }

    @Override
    public PythonManagingContainer getManagingContainer() {
        return domain;
    }

    public int getStartOrder() {
        return startOrder;
    }

    public int getStopOrder() {
        return stopOrder;
    }

    public List<Step> controlTaskDispatch(String name, Map<String,String> args) {
        if (name.equals(STOP_CONTROL_TASK)) {
            return getStopSteps(1);
        }

        if (name.equals(START_CONTROL_TASK)) {
            return getStartSteps(1);
        }
        return ControlTaskDelegate.dispatch(name, args, this, this);
    }

    @Override
    public Host getHost() {
        return domain.getHost();
    }

    public List<Step> getStartSteps(final int order) {
        return newArrayList(concat(transform(getServers(), new Function<Server, List<Step>>() {
            @Override
            public List<Step> apply(Server input) {
                return input.getStartSteps(order);
            }
        })));
    }

    public List<Step> getStopSteps(final int order) {
        return newArrayList(concat(transform(getServers(), new Function<Server, List<Step>>() {
            @Override
            public List<Step> apply(Server input) {
                return input.getStopSteps(order);
            }
        })));
    }
}
