package com.xebialabs.deployit.plugin.wls.contributor;

import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedArtifact;
import com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedSharedLibrary;

import static com.google.common.base.Predicates.and;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;
import static com.xebialabs.deployit.plugin.wls.contributor.Predicates2.deltaOf;
import static com.xebialabs.deployit.plugin.wls.contributor.Predicates2.extractDeployed;
import static com.xebialabs.deployit.plugin.wls.contributor.Predicates2.instanceOf;
import static com.xebialabs.deployit.plugin.wls.contributor.Predicates2.operationIs;

public class RefreshSharedLibRuntimeDependencies {

    private static final Logger logger = LoggerFactory.getLogger(RefreshSharedLibRuntimeDependencies.class);

    @Contributor
    public static void handleRuntimeDepenencies(final Deltas deltas, DeploymentPlanningContext ctx) {

        Iterable<Deployed<?, ?>> modifiedSharedLibs = transform(
                filter(deltas.getDeltas(), and(operationIs(MODIFY), deltaOf(Type.valueOf(ExtensibleDeployedSharedLibrary.class)))), extractDeployed()
        );

        for (final Deployed<?, ?> sharedLib : modifiedSharedLibs) {
            final List<Deployed> dependentArtifacts = searchDependentArtifacts(sharedLib, ctx);
            logger.debug("dependentArtifacts: " + dependentArtifacts);

            final Iterable<Deployed> runtimeDependenciesDeployedArtifacts = filter(dependentArtifacts, new Predicate<Deployed>() {
                @Override
                public boolean apply(final Deployed deployedArtifact) {
                    if (isArtifactInCurrentDeployment(deltas, deployedArtifact)) {
                        return false;
                    }
                    final List<ConfigurationItem> sharedLibraries = deployedArtifact.getProperty("sharedLibraries");
                    return sharedLibraries.contains(sharedLib);
                }
            });

            for (Deployed deployed : runtimeDependenciesDeployedArtifacts) {
                logger.debug("%s(%s) Operation.MODIFY", deployed.getId(), deployed.getType());
                ExtensibleDeployedArtifact artifact = ctx.getRepository().read(deployed.getId());

                Delta destroyArtifact = new RuntimeDelta(Operation.DESTROY, artifact, null);
                artifact.stopApplication(ctx, destroyArtifact);
                artifact.destroyArtifact(ctx, destroyArtifact);

                Delta createArtifact = new RuntimeDelta(Operation.CREATE, null, artifact);
                artifact.deployArtifact(ctx, createArtifact);
                artifact.startApplication(ctx, createArtifact);
            }
        }
    }

    private static boolean isArtifactInCurrentDeployment(Deltas deltas, final Deployed dependentArtifact) {
        final Iterable<Deployed<?, ?>> deploymentArtifacts = filter(transform(deltas.getDeltas(), extractDeployed()), instanceOf(Type.valueOf(ExtensibleDeployedArtifact.class)));

        return Iterables.any(deploymentArtifacts, new Predicate<Deployed>() {
            @Override
            public boolean apply(final Deployed deploymentArtifact) {
                return dependentArtifact.getId().equals(deploymentArtifact.getId());
            }
        });
    }

    @SuppressWarnings("unchecked")
    private static List<Deployed> searchDependentArtifacts(final Deployed deployed, final DeploymentPlanningContext context) {
        final ImmutableList.Builder builder = new ImmutableList.Builder();
        final Repository repository = context.getRepository();
        for (String type : (List<String>) deployed.getProperty("runtimeDependenciesTypes")) {
            builder.addAll(repository.search(Type.valueOf(type), deployed.getContainer().getId()));
        }
        return builder.build();
    }

    @SuppressWarnings("serial")
    public static class RuntimeDelta implements Delta {
        private Operation operation;

        @SuppressWarnings("rawtypes")
        private Deployed previous;

        @SuppressWarnings("rawtypes")
        private Deployed deployed;

        @SuppressWarnings("rawtypes")
        public RuntimeDelta(Operation operation, Deployed previous, Deployed deployed) {
            super();
            this.operation = operation;
            this.previous = previous;
            this.deployed = deployed;
        }

        @Override
        public Operation getOperation() {
            return operation;
        }

        @Override
        public Deployed<?, ?> getPrevious() {
            return previous;
        }

        @Override
        public Deployed<?, ?> getDeployed() {
            return deployed;
        }

        @Override
        public List<String> getIntermediateCheckpoints() {
            return new ArrayList<>();
        }
    }
}
