/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.container;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.inspection.InspectionProperty;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.python.ControlTaskDelegate;
import com.xebialabs.deployit.plugin.python.PythonInspectionStep;
import com.xebialabs.deployit.plugin.python.PythonManagedContainer;
import com.xebialabs.deployit.plugin.python.PythonManagingContainer;
import com.xebialabs.deployit.plugin.wls.DeploymentMode;
import com.xebialabs.deployit.plugin.wls.freemarker.TemplateEvaluator;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.wls.container.DelegatedInspectionHelper.discoverDeployeds;

/**
 * A domain in a WebLogic WLS installation.
 *
 * @see Cluster
 * @see Server
 */
@SuppressWarnings("serial")
@Metadata(
        root = Metadata.ConfigurationItemRoot.INFRASTRUCTURE,
        description = "WebLogic Domain which is a collection of WebLogic Server instances that is managed by a single Administration Server",
        inspectable = true)
public class Domain extends BaseContainer implements PythonManagingContainer, PythonManagedContainer, WlsContainer {

    @Property(required = true, label = "Host", description = "The host that runs the admin server")
    @InspectionProperty
    private Host host;

    @Property(required = true, defaultValue = "WEBLOGIC_10", label = "Version", description = "Version of Oracle WebLogic Server")
    @InspectionProperty
    private Version version;

    @Property(required = true, label = "WebLogic home", description = "The location of the WebLogic Server installation")
    @InspectionProperty
    private String wlHome;

    @Property(required = false, label = "WebLogic domain home", description = "The location of the WebLogic domain. Defaults to <wlHome>/../user_projects/domains/<name> (Unix) or <wlHome>\\..\\user_projects\\domains\\<name> (Windows)")
    private String domainHome;

    @Property(required = false, label = "WLST path", description = "Location of the wlst binary. Defaults to <wlHome>/common/bin/wlst.sh (Unix) or <wlHome>\\common\\bin\\wlst.cmd (Windows)")
    @InspectionProperty(required = false)
    private String wlstPath;

    @Property(required = false, hidden = true, description = "Path containing the custom wlst templates: wlst.sh.ftl (Unix) and wlst.cmd.ftl (Windows)")
    private String customWlstTemplatePath;

    @Property(defaultValue = "t3", required = true, label = "Administrative server protocol", description = "Protocol to be used by the AdminServer for this domain")
    @InspectionProperty(required = false)
    private WlsProtocol protocol;

    @Property(required = false, label = "Administrative server host", description = "Host to connect to for WLST")
    @InspectionProperty(required = false)
    private String hostname;

    @Property(required = true, defaultValue = "7001", label = "Administrative server port", description = "Port to be used by the AdminServer for this domain")
    @InspectionProperty(required = false)
    private int port;

    @Property(required = true, label = "Administrative username", description = "Username which is used to login to the WebLogic Domain.")
    @InspectionProperty
    private String username;

    @Property(required = true, label = "Administrative password", password = true, description = "Password which is used to login to the WebLogic Domain.")
    @InspectionProperty
    private String password;

    @Property(required = true, defaultValue = "AdminServer", description = "The name of the admin server")
    private String adminServerName;

    @Property(required = true, defaultValue = "NodeManager", label = "Start Mode", description = "Tells how a managed server is start and stop, default is NodeManager, others are Script or Windows Service")
    private StartMode startMode;

    @Property(required = false, defaultValue = "true", hidden = true, label = "Run with daemon", description = "Set to true to execute commands with the Python daemon")
    private boolean runWithDaemon;

    @Property(required = false, defaultValue = "SINGLE_TARGET", label = "Deployment mode", category = "Deployment", description = "Specify if deploy, undeploy, start and stop operations should be batched and target multiple servers as part of one WLST command. Valid values: 'SINGLE_TARGET' (default, issues a separate call to each target server), 'MULTI_TARGET' (issues one call for multiple target servers by setting the WLST targets options).")
    private DeploymentMode deploymentMode;

    @Property(description = "WebLogic servers belonging to domain", category = "Topology", asContainment = true)
    private Set<Server> servers = Sets.newHashSet();

    @Property(description = "WebLogic clusters belonging to domain", category = "Topology", asContainment = true)
    private Set<Cluster> clusters = Sets.newHashSet();

    @Property(required = true, defaultValue = "80", hidden = true)
    private int startOrder;

    @Property(required = true, defaultValue = "20", hidden = true)
    private int stopOrder;

    @Property(required = true, defaultValue = "100", hidden = true, description = "Order used to trigger the clean of the JMS Modules of the domain")
    private int destroyJmsModuleOrder;

    @Property(hidden = true, defaultValue = "wls.FilePersistentStore")
    private Set<String> deployedsToDiscover = newHashSet();

    @Override
    public PythonManagingContainer getManagingContainer() {
        return this;
    }

    @Override
    public CmdLine getScriptCommandLine(OverthereFile script) {
        if(Strings.isNullOrEmpty(customWlstTemplatePath)) {
            return getStandardScriptCommandLine(script, "wls");
        } else {
            return getScriptCommandLineForTemplate(script, customWlstTemplatePath);
        }
    }

    @Inspect
    public void inspect(InspectionContext ctx) {
        Map<String, Object> pythonVars = newHashMap();
        pythonVars.put("container", this);
        ctx.addStep(new PythonInspectionStep(this, this, "wls/container/inspect-domain.py", pythonVars, "Inspect " + this));
        Iterable<Type> typesToDiscovery = Iterables.transform(deployedsToDiscover, new Function<String, Type>() {
            @Override
            public Type apply(String input) {
                return Type.valueOf(input);
            }
        });
        discoverDeployeds(this, ctx, Lists.newArrayList(typesToDiscovery));
    }

    public List<Step> controlTaskDispatch(String name, Map<String, String> args) {
        return ControlTaskDelegate.dispatch(name, args, this, this);
    }

    @Override
    public Host getHost() {
        return host;
    }

    public void setHost(Host activeHost) {
        this.host = activeHost;
    }

    public Set<String> getDeployedsToDiscover() {
        return deployedsToDiscover;
    }

    public void setDeployedsToDiscover(final Set<String> deployedsToDiscover) {
        this.deployedsToDiscover = deployedsToDiscover;
    }

    public Version getVersion() {
        return version;
    }

    public void setVersion(Version version) {
        this.version = version;
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public String getWlHome() {
        return wlHome;
    }

    public void setWlHome(String wlHome) {
        this.wlHome = wlHome;
    }

    public String getDomainHome() {
        return (isNullOrEmpty(domainHome) ? getDefaultDomainHome() : domainHome);
    }

    private String getDefaultDomainHome() {
        final String fileSeparator = host.getOs().getFileSeparator();
        return wlHome.substring(0, wlHome.lastIndexOf(fileSeparator)) + fileSeparator + "user_projects" + fileSeparator + "domains" + fileSeparator + getName();
    }

    public void setDomainHome(String domainHome) {
        this.domainHome = domainHome;
    }

    public String getWlstPath() {
        return (isNullOrEmpty(wlstPath) ? getDefaultWlstPath() : wlstPath);
    }

    private String getDefaultWlstPath() {
        String fileSeparator = this.getHost().getOs().getFileSeparator();
        String wlstExtension;
        if (getHost().getOs() == OperatingSystemFamily.WINDOWS) {
            wlstExtension = ".cmd";
        } else {
            wlstExtension = ".sh";
        }
        return wlHome + fileSeparator + "common" + fileSeparator + "bin" + fileSeparator + "wlst" + wlstExtension;
    }

    public void setWlstPath(String wlstPath) {
        this.wlstPath = wlstPath;
    }

    public String getAdminServerName() {
        return adminServerName;
    }

    public void setAdminServerName(String adminServerName) {
        this.adminServerName = adminServerName;
    }

    public void setUseNodeManager(boolean useNodeManager) {
        this.startMode = StartMode.NodeManager;
    }

    public StartMode getStartMode() {
        return startMode;
    }

    public void setStartMode(StartMode startMode) {
        this.startMode = startMode;
    }

    public Set<Cluster> getClusters() {
        return clusters;
    }

    public void addClusters(Cluster cluster) {
        clusters.add(cluster);
    }

    @Override
    public String getRuntimePath() {
        return "wls/runtime";
    }

    @Override
    public boolean runWithDaemon() {
        return runWithDaemon;
    }

    public void setRunWithDaemon(boolean runWithDaemon) {
        this.runWithDaemon = runWithDaemon;
    }

    public DeploymentMode getDeploymentMode() {
        return deploymentMode;
    }

    public void setDeploymentMode(DeploymentMode deploymentMode) {
        this.deploymentMode = deploymentMode;
    }

    @Override
    public Domain getDomain() {
        return this;
    }

    @Override
    public String getObjectName() {
        return "com.bea:Name=" + getName() + ",Type=Domain";
    }

    @Override
    public Set<Host> getHosts() {
        return Sets.newHashSet(transform(servers, new Function<Server, Host>() {
            @Override
            public Host apply(final Server host) {
                return host.getHost();
            }
        }));
    }

    @Override
    public Set<Server> getServers() {
        return servers;
    }

    @Override
    public int getStopOrder() {
        return stopOrder;
    }

    @Override
    public int getStartOrder() {
        return startOrder;
    }

    @Override
    public List<Step> getStartSteps(final int order) {
        return newArrayList(concat(transform(servers, new Function<Server, List<Step>>() {
            @Override
            public List<Step> apply(final Server server) {
                return server.getStartSteps(order);
            }
        })));
    }

    @Override
    public List<Step> getStopSteps(final int order) {
        return newArrayList(concat(transform(servers, new Function<Server, List<Step>>() {
            @Override
            public List<Step> apply(final Server server) {
                return server.getStopSteps(order);
            }
        })));
    }

    public int getDestroyJmsModuleOrder() {
        return destroyJmsModuleOrder;
    }

    protected CmdLine getScriptCommandLineForTemplate(OverthereFile script, String template) {
        final CmdLine cmdLine = getCmdLine();
        cmdLine.addArgument(getRemoteTemplate(script, template).getPath());
        return cmdLine;
    }

    protected void uploadPasswordFile(OverthereFile script, String template) {
        final OverthereFile t = script.getParentFile().getFile("wlst.pass");
        TemplateEvaluator.evaluateTemplate(t, template + "/wlst.pass.ftl",
                ImmutableMap.<String, Object> of("container", this, "script", script.getPath()));
    }

    private CmdLine getStandardScriptCommandLine(OverthereFile script, String template) {
        CmdLine cmdLine = getScriptCommandLineForTemplate(script, template);
        uploadPasswordFile(script, template);
        return cmdLine;
    }

    private OverthereFile getRemoteTemplate(OverthereFile script, String template) {

        final String templatePath;
        final OverthereFile remoteWlst;

        if (isUnix()) {
            templatePath = template + "/wlst.sh.ftl";
            remoteWlst = script.getParentFile().getFile("wlst.sh");

        } else {
            templatePath = template + "/wlst.cmd.ftl";
            remoteWlst = script.getParentFile().getFile("wlst.cmd");
        }

        TemplateEvaluator.evaluateTemplate(remoteWlst, templatePath, ImmutableMap.<String, Object> of("container", this, "script", script.getPath()));
        return remoteWlst;
    }

    private CmdLine getCmdLine() {
        final CmdLine cmdLine = new CmdLine();
        if (isUnix())
            cmdLine.addArgument("sh");
        return cmdLine;
    }

    private boolean isUnix() {
        return getHost().getOs() == OperatingSystemFamily.UNIX ||
                getHost().getOs() == OperatingSystemFamily.ZOS;
    }
}
