package com.xebialabs.deployit.plugin.wls.deployed;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.validation.Regex;
import com.xebialabs.deployit.plugin.python.PythonDeploymentStep;
import com.xebialabs.deployit.plugin.python.PythonManagedContainer;
import com.xebialabs.deployit.plugin.python.PythonManagedDeployed;

import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "Base deployed of all WLS deployeds.")
public class BaseWlsDeployed<D extends Deployable, C extends PythonManagedContainer> extends PythonManagedDeployed<D, C> {
    public static final String STEP_OPTION_NONE = "none";

    public static final String STEP_OPTION_UPLOAD_ARTIFACT_DATA = "uploadArtifactData";

    public static final String STEP_OPTION_EXPOSE_DEPLOYED_APPLICATION = "exposeDeployedApplication";

    public static final String STEP_OPTION_EXPOSE_PREVIOUS_DEPLOYED = "exposePreviousDeployed";

    public static final String STEP_OPTIONS_REGEX = "(" + STEP_OPTION_NONE + "|" + STEP_OPTION_UPLOAD_ARTIFACT_DATA
            + "|" + STEP_OPTION_EXPOSE_DEPLOYED_APPLICATION + "|" + STEP_OPTION_EXPOSE_PREVIOUS_DEPLOYED + ")";

    public static final String STEP_OPTIONS_DESCRIPTION_SUFFIX = " (1 or more of: " + STEP_OPTION_NONE + "," + STEP_OPTION_UPLOAD_ARTIFACT_DATA
            + "," + STEP_OPTION_EXPOSE_DEPLOYED_APPLICATION + "," + STEP_OPTION_EXPOSE_PREVIOUS_DEPLOYED + ").";

    public static final String STEP_OPTIONS_DEFAULT = STEP_OPTION_UPLOAD_ARTIFACT_DATA;

    @Property(hidden = true, defaultValue = STEP_OPTIONS_DEFAULT, description = "Options for the create step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    protected Set<String> createOptions;

    @Property(hidden = true, defaultValue = STEP_OPTIONS_DEFAULT, description = "Options for the modify step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    protected Set<String> modifyOptions;

    @Property(hidden = true, defaultValue = STEP_OPTIONS_DEFAULT, description = "Options for the destroy step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    protected Set<String> destroyOptions;

    @Property(hidden = true, defaultValue = STEP_OPTIONS_DEFAULT, description = "Options for the noop step" + STEP_OPTIONS_DESCRIPTION_SUFFIX)
    @Regex(pattern = STEP_OPTIONS_REGEX)
    protected Set<String> noopOptions;

    public boolean addStep(DeploymentPlanningContext ctx, Delta delta, int order, String scriptPath, String verb, Set<String> options) {
        return addStep(ctx, delta, null, order, scriptPath, verb, options);
    }

    public boolean addStepWithCheckpoint(DeploymentPlanningContext ctx, Delta delta, Operation operation, int order, String scriptPath, String verb, Set<String> options) {
        Preconditions.checkNotNull(operation, "operation is null");
        return addStep(ctx, delta, operation, order, scriptPath, verb, options);
    }

    protected boolean addStep(DeploymentPlanningContext ctx, Delta delta, Operation operation, int order, String scriptPath, String verb, Set<String> options) {
        if (!Strings.isNullOrEmpty(scriptPath)) {
            PythonDeploymentStep step = createStep(ctx, delta, order, scriptPath, verb, options);
            if (operation == null) {
                ctx.addStep(step);
            } else {
                ctx.addStepWithCheckpoint(step, delta, operation);
            }
            return true;
        } else {
            return false;
        }
    }

    protected PythonDeploymentStep createStep(DeploymentPlanningContext ctx, Delta delta, int order, String scriptPath, String verb, Set<String> options) {
        Map<String, Object> pythonVars = getPythonVars(delta.getPrevious(), ctx.getDeployedApplication(), options);
        PythonDeploymentStep step = new PythonDeploymentStep(order, getContainer().getManagingContainer(), scriptPath, pythonVars, getDescription(verb));
        step.setUploadArtifactData(options.contains(STEP_OPTION_UPLOAD_ARTIFACT_DATA));
        return step;
    }

    protected Map<String, Object> getPythonVars(Deployed<?, ?> previous, DeployedApplication deployedApplication, Set<String> options) {
        Map<String, Object> pythonVars = newHashMap();
        pythonVars.put("deployed", this);
        if (options.contains(STEP_OPTION_EXPOSE_PREVIOUS_DEPLOYED)) {
            checkNotNull(previous, "previousDeployed is null");
            pythonVars.put("previousDeployed", previous);
        }
        if (options.contains(STEP_OPTION_EXPOSE_DEPLOYED_APPLICATION) || getExposeDeployedApplication()) {
            checkNotNull(deployedApplication, "deployedApplication is null");
            pythonVars.put("deployedApplication", deployedApplication);
        }
        return pythonVars;
    }

}
