package com.crossingchannels.portal.websphere.specification.jee5;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.jar.JarFile;

import org.apache.commons.lang.Validate;

import com.crossingchannels.portal.websphere.specification.jsr286.PortletDescriptor;

/**
 * Represents an Web Archive.
 * 
 * @author FWiegerinck
 */
public class WebArchive {

    /**
     * Load archive from disk
     * 
     * @param location
     *            Location of the archive.
     * @return Instance of WebArchive.
     * @throws IOException
     */
    public static WebArchive fromDisk(final String location) throws IOException {
        Validate.notNull(location);

        final File fileRef = new File(location);
        // Ensure file is valid
        if (!fileRef.exists() || !fileRef.canRead()) {
            throw new FileNotFoundException(String.format("Unable to read file [%s]", location));
        }

        return new WebArchive(fileRef);
    }

    private final String archiveName;

    private final File archiveOnDisk;

    private WebDescriptor webDescriptor = null;

    private PortletDescriptor portletDescriptor = null;

    private WebArchive(final File archive) throws IOException {

        // Save information
        this.archiveOnDisk = archive;
        this.archiveName = archive.getName();

    }

    public String getArchiveName() {
    	return this.archiveName;
    }
    
    public WebDescriptor getDescriptor() throws IllegalDescriptorFoundException, IOException {
        if (this.webDescriptor == null) {
            this.webDescriptor = this.loadWebDescriptor();
        }

        return this.webDescriptor;
    }

    public PortletDescriptor getPortletDescriptor() throws IllegalDescriptorFoundException, IOException {
        if (this.portletDescriptor == null) {
            this.portletDescriptor = this.loadPortletDescriptor();
        }

        return this.portletDescriptor;
    }

    /**
     * @return The WebDescriptor from the archive
     * @throws IllegalDescriptorFoundException
     * @throws IOException
     */
    private WebDescriptor loadWebDescriptor() throws IllegalDescriptorFoundException, IOException {
        final JarFile warFile = new JarFile(this.archiveOnDisk);
        final InputStream descriptor = warFile.getInputStream(warFile.getJarEntry("WEB-INF/web.xml"));
        try {
            return WebDescriptor.fromInputStream(descriptor);
        } finally {
            warFile.close();
        }
    }

    /**
     * @return The WebDescriptor from the archive
     * @throws IllegalDescriptorFoundException
     * @throws IOException
     */
    private PortletDescriptor loadPortletDescriptor() throws IllegalDescriptorFoundException, IOException {
        final JarFile warFile = new JarFile(this.archiveOnDisk);
        final InputStream descriptor = warFile.getInputStream(warFile.getJarEntry("WEB-INF/portlet.xml"));
        try {
            return PortletDescriptor.fromInputStream(descriptor);
        } finally {
            warFile.close();
        }
    }

}
