package ext.deployit.com.crossingchannels.portal.websphere.ci.generic.deployed;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployedArtifact;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalFile;

import ext.deployit.com.crossingchannels.portal.websphere.ci.generic.container.WpContainer;
import ext.deployit.com.crossingchannels.portal.websphere.ci.generic.deployable.DeployableXmlAccessScript;
import ext.deployit.com.crossingchannels.portal.websphere.ci.generic.step.ExecuteXmlAccessScriptStep;

/**
 * Deployed CI identifying a deployed XML access script resource.
 * 
 * @author fwiegerinck
 */
@Metadata(virtual = false, description = "Deployed XML Access script")
@SuppressWarnings("serial")
public class DeployedXmlAccessScript extends BaseDeployedArtifact<DeployableXmlAccessScript, WpContainer> {

    private static final Logger LOGGER = LoggerFactory.getLogger(DeployedXmlAccessScript.class);

    @Property(hidden = false, label = "Create script", description = "XML Access Script used to install. Also used during update if no update script is specified.", category = "Advanced")
    private String createScript;

    @Property(hidden = false, label = "Destroy script", description = "XML Access Script used to uninstall. Also used during update if no update script is specified.", category = "Advanced", required = false)
    private String destroyScript;

    @Property(hidden = false, label = "Modify script", description = "XML Access Script used to update from a previous version.", category = "Advanced", required = false)
    private String modifyScript;

    @Property(hidden = true, label = "Order of creation", defaultValue = "79")
    private int createOrder;

    @Property(hidden = true, label = "Order of destruction", defaultValue = "21", required = false)
    private int destroyOrder;

    @Property(hidden = true, label = "Order of modification", defaultValue = "21", required = false)
    private int modifyOrder;

	public String getCreateScript() {
		return createScript;
	}

	public void setCreateScript(String create) {
		this.createScript = create;
	}

	public String getDestroyScript() {
		return destroyScript;
	}

	public void setDestroyScript(String destroy) {
		this.destroyScript = destroy;
	}

	public String getModifyScript() {
		return modifyScript;
	}

	public void setModifyScript(String modify) {
		this.modifyScript = modify;
	}

	public int getCreateOrder() {
		return createOrder;
	}

	public void setCreateOrder(int createOrder) {
		this.createOrder = createOrder;
	}

	public int getDestroyOrder() {
		return destroyOrder;
	}

	public void setDestroyOrder(int destroyOrder) {
		this.destroyOrder = destroyOrder;
	}

	public int getModifyOrder() {
		return modifyOrder;
	}

	public void setModifyOrder(int modifyOrder) {
		this.modifyOrder = modifyOrder;
	}  

	@Create
    public void createResource(final DeploymentPlanningContext ctx) throws IOException {
        if (StringUtils.isNotEmpty(this.createScript)) {
            this.addStep(ctx, this.createScript, this.createOrder);
        }
    }

    @Modify
    public void modifyResource(final DeploymentPlanningContext ctx) throws IOException {
        if (StringUtils.isNotEmpty(this.modifyScript)) {
            this.addStep(ctx, this.modifyScript, this.modifyOrder);
        } else {
            this.destroyResource(ctx);
            this.createResource(ctx);
        }

    }

    @Destroy
    public void destroyResource(final DeploymentPlanningContext ctx) throws IOException {
        if (StringUtils.isNotEmpty(this.destroyScript)) {
            this.addStep(ctx, this.destroyScript, this.destroyOrder);
        }
    }

    protected void addStep(final DeploymentPlanningContext ctx, final String script, final int order) throws IOException {
        final String locatedFile = this.locateFile(script);
        if ( StringUtils.isNotEmpty(locatedFile)) {
        	ctx.addStep(new ExecuteXmlAccessScriptStep(locatedFile, order, this));
        } else {
        	DeployedXmlAccessScript.LOGGER.error("Unable to locate script: {}", script);
        }
    }

    @ControlTask(description = "Reinstall the XML Access script")
    public List<? extends Step> reinstall() throws IOException {
        return Collections.singletonList(new ExecuteXmlAccessScriptStep(this.createScript, this.createOrder, this));
    }

    private String locateFile(final String filter) {
        Validate.notEmpty(filter);

        final List<OverthereFile> files = this.getFile().listFiles();
        for (final OverthereFile file : files) {
            if (filter.equals(file.getName()) && (file instanceof LocalFile)) {

                final String filename = file.getPath();

                DeployedXmlAccessScript.LOGGER.debug("Found file [{}]", filename);

                return filename;
            }
        }
        return null;
    }
}
