package ext.deployit.com.crossingchannels.portal.websphere.ci.policy.contributor;

import java.text.MessageFormat;

import javax.xml.xpath.XPathExpressionException;

import com.crossingchannels.portal.websphere.ci.base.contributor.AbstractStepsContributor;
import com.crossingchannels.portal.websphere.ci.policy.step.PolicyDeploymentStep;
import com.crossingchannels.portal.websphere.util.XmlUtil;
import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployableFileArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployedArtifact;

import ext.deployit.com.crossingchannels.portal.websphere.ci.generic.container.WpRuntimeContainer;
import ext.deployit.com.crossingchannels.portal.websphere.ci.generic.step.ExecuteXmlAccessConfigurationItemStep;

/**
 * Contributor to add steps when policy step is added.
 * 
 * @author fwiegerinck
 * @since 1.1
 * @version 1.1
 */
public class PolicyStepsContributor extends AbstractStepsContributor<BaseDeployedArtifact<BaseDeployableFileArtifact, WpRuntimeContainer>> {

    /**
     * Resolve type definition for a policy.
     */
    private static final Type TYPE_POLICY = Type.valueOf("wp.Policy");

    /**
     * Resolve type definition for a portlet container.
     */
    private static final Type TYPE_WPRUNTIMECONTAINER = Type.valueOf("wp.WpRuntimeContainer");

    /**
     * Define name of the CI-property containing the register XML Access Script.
     */
    private static final String PROPERTY_REGISTER = "registerScript";

    /**
     * Define name of the CI-property containing the unregister XML Access Script.
     */
    private static final String PROPERTY_UNREGISTER = "unregisterScript";

    /**
     * Define name of the CI-property containing the order to run the register script.
     */
    private static final String PROPERTY_REGISTER_ORDER = "registerScriptOrder";

    /**
     * Define order used to arrange the step to unregister policies.
     */
    private static final String PROPERTY_UNREGISTER_ORDER = "unregisterScriptOrder";

	/**
	 * Define name of property containing the policy name
	 */
	private static final String PROPERTY_POLICY_NAME = "policyName";

	/**
	 * Define expression to use to get the name of the policy
	 */
	private static final String POLICYNAME_XPATH_EXPRESSION = "/policyList/policy/policyValue[@Name='PznType']/value";

	/**
     * Initialize a new installation steps contributor for Policy.
     */
    public PolicyStepsContributor() {
        super(PolicyStepsContributor.TYPE_POLICY, TYPE_WPRUNTIMECONTAINER);
    }

    // Override to trigger autodetection
    @Override
    @Contributor
    public void contributePortletSteps(final Deltas deltas, final DeploymentPlanningContext context) {
        super.contributePortletSteps(deltas, context);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void install(final BaseDeployedArtifact<BaseDeployableFileArtifact, WpRuntimeContainer> artifactToInstall, final DeploymentPlanningContext context, final Delta delta) {
    	// Determine name of the policy to install and set the property
    	this.lookupPolicyName(artifactToInstall);

    	//add required step (different from uninstall since it needs to copy a artifact first)
    	context.addStep(
    			new PolicyDeploymentStep(
    					artifactToInstall.<String> getProperty(PolicyStepsContributor.PROPERTY_REGISTER), 
    					artifactToInstall, 
    					artifactToInstall.<Integer> getProperty(PolicyStepsContributor.PROPERTY_REGISTER_ORDER)
    				)
    		);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void update(final BaseDeployedArtifact<BaseDeployableFileArtifact, WpRuntimeContainer> newArtifact, final BaseDeployedArtifact<BaseDeployableFileArtifact, WpRuntimeContainer> oldArtifact, final DeploymentPlanningContext context, final Delta delta) {
    	// Installs and updates are the same
        this.install(newArtifact, context, delta);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void uninstall(final BaseDeployedArtifact<BaseDeployableFileArtifact, WpRuntimeContainer> artifactToUninstall, final DeploymentPlanningContext context, final Delta delta) {
    	// Determine name of the policy to uninstall and set the property
    	this.lookupPolicyName(artifactToUninstall);
    	
    	// add required step (different from install since no copy of artifact is needed)
        context.addStep( new ExecuteXmlAccessConfigurationItemStep(artifactToUninstall.<String> getProperty(PolicyStepsContributor.PROPERTY_UNREGISTER), artifactToUninstall,
                artifactToUninstall.<Integer> getProperty(PolicyStepsContributor.PROPERTY_UNREGISTER_ORDER)));
    }
    
    /**
     * Lookup policy name using XPath query.
     * 
     * @param artifact Artifact containing the policy XML.
     */
    private void lookupPolicyName( final BaseDeployedArtifact<BaseDeployableFileArtifact, WpRuntimeContainer> artifact) {
    	try {
			final String name = XmlUtil.getValue(POLICYNAME_XPATH_EXPRESSION, artifact.getFile().getInputStream());
			artifact.setProperty(PolicyStepsContributor.PROPERTY_POLICY_NAME, name);
		} catch (XPathExpressionException e) {
            throw new IllegalArgumentException(MessageFormat.format("Missing PznType in policy xml for CI {0}", artifact.getId()));
		}
    }

}
