package com.crossingchannels.portal.websphere.ci.base.step;

import java.io.IOException;
import java.io.OutputStream;
import java.text.MessageFormat;
import java.util.Map;

import org.apache.commons.lang.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crossingchannels.portal.websphere.executor.ExecutorFactory;
import com.crossingchannels.portal.websphere.executor.XmlAccessExecutionException;
import com.crossingchannels.portal.websphere.executor.XmlAccessExecutor;
import com.crossingchannels.portal.websphere.specification.wp.dom.Request;
import com.crossingchannels.portal.websphere.template.TemplateParseException;
import com.crossingchannels.portal.websphere.template.TemplateParser;
import com.google.common.io.Closeables;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalConnection;

import ext.deployit.com.crossingchannels.portal.websphere.ci.generic.container.WpContainer;


/**
 * Provides common functionality to execute an XML access script
 *
 * @author fwiegerinck
 */
@SuppressWarnings("serial")
public abstract class AbstractXmlAccessDeploymentStep extends AbstractDeploymentStep {

    /**
     * Define logger for this class.
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractXmlAccessDeploymentStep.class);

    /**
     * @param description
     * @param order
     */
    public AbstractXmlAccessDeploymentStep(final String description, final int order) {
        super(description, order);
    }

    /**
     * Return instance of the template parser.
     *
     * @return Valid instance of a parser.
     */
    protected abstract TemplateParser getTemplateParser();

    /**
     * Execute the xml access script using the supplied CI and parse context
     *
     * @param ctx
     *            The deployment context use to log information.
     * @param xmlAccessScript
     *            Location of the XML access script template.
     * @param deployedCi
     *            The CI associated with the XML access script.
     * @param parseContext
     *            The context used to parse the XML access template.
     * @return TRUE if successfully executed, FALSE otherwise.
     */
    protected boolean executeXmlAccess(final ExecutionContext ctx, final String xmlAccessScript, final Deployed<? extends Deployable, ? extends WpContainer> deployedCi,
            final Map<String, ?> parseContext) {

        // Preconditions
        Validate.notNull(ctx);
        Validate.notNull(xmlAccessScript);
        Validate.notNull(deployedCi);
        Validate.notNull(parseContext);

        // Log info to user
        ctx.logOutput("Start generating XML Access for execution.");

        // Parse template first
        final OverthereFile parsedScript = this.parseScript(xmlAccessScript, deployedCi, parseContext);

        // Log info to user
        ctx.logOutput("Finished generating XML Access for execution.");

        // Get executor
        final XmlAccessExecutor executor = ExecutorFactory.INSTANCE.createXmlAccessExecutor(deployedCi.getContainer());

        try {
            // Do execute
            final Request returnedDom = executor.executeXmlAccess(ctx, parsedScript, xmlAccessScript);

            // Test for success
            if ((returnedDom != null) && executor.isSuccessfullyExecuted(returnedDom)) {
                ctx.logOutput(MessageFormat.format("\nExecuted script ({0}) for CI({1}) successfully!", xmlAccessScript, deployedCi.getId()));
                return true;
            } else {
                ctx.logError(MessageFormat.format("\nUnable to execute script ({0}) for CI ({1})", xmlAccessScript, deployedCi.getId()));
                return false;
            }

        } catch (final XmlAccessExecutionException xaee) {
            ctx.logError(MessageFormat.format("\nUnable to execute script ({0}) for CI ({1})", xmlAccessScript, deployedCi.getId()), xaee);
            return false;
        }

    }

    /**
     * Parse the script provided using the <code>deployed CI</code> as subject.
     *
     * @param xmlAccessScript
     *            The XML Access script to parse.
     * @param deployedCi
     *            The CI to add to the freemarker parsing context.
     * @param parseContext
     *            The context used to parse the freemarker template
     * @return Reference to a local temporary file with the <strong>parsed</strong> templated.
     */
    private OverthereFile parseScript(final String xmlAccessScript, final Deployed<? extends Deployable, ? extends Container> deployedCi, final Map<String, ?> parseContext) {

        // Get temporary file
        final OverthereConnection local = LocalConnection.getLocalConnection();
        final OverthereFile temporaryFile = local.getTempFile("execute_xmlaccess", ".xml");

        // Log temp file
        AbstractXmlAccessDeploymentStep.LOGGER.debug("Use temporary file {} to parse template", temporaryFile.toString());

        try (final OutputStream temporaryFileOutputStream = temporaryFile.getOutputStream()) {
            // Do parse
            this.getTemplateParser().parse(xmlAccessScript, deployedCi, parseContext, temporaryFileOutputStream);
        } catch (final TemplateParseException tpe) {
            throw new RuntimeException("Unable to parse template", tpe);
        } catch (IOException e) {
            throw new RuntimeException("IO Exception opening temporary file", e);
        }

        return temporaryFile;
    }

}
