package com.crossingchannels.portal.websphere.executor;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.text.MessageFormat;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.SAXException;

import com.crossingchannels.portal.websphere.specification.wp.dom.ObjectFactory;
import com.crossingchannels.portal.websphere.specification.wp.dom.Request;
import com.crossingchannels.portal.websphere.specification.wp.dom.StatusInfo;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.overthere.DefaultProcessOutputHandler;
import com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalConnection;

import ext.deployit.com.crossingchannels.portal.websphere.ci.generic.container.WpContainer;
import ext.deployit.com.crossingchannels.portal.websphere.ci.generic.container.WpRuntimeContainer;

/**
 * Execute XmlAccess command on (remote) host.
 * 
 * @author fwiegerinck
 */
public class XmlAccessExecutor implements Executor {

    /**
     * Define logger for this class.
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(XmlAccessExecutor.class);

    /**
     * Define logger used to log the executed XML access scripts
     */
    private static final Logger SCRIPTLOGGER = LoggerFactory.getLogger("com.crossingchannels.portal.websphere.xmlaccess.scripts");
    
    /**
     * Define name of XSD.
     */
    public static final String XSD_FILENAME = "PortalConfig_6.1.0.xsd";

    /**
     * Define URL to XSD_LOCATION.
     */
    public static final URL XSD_LOCATION = XmlAccessExecutor.class.getResource("/META-INF/xsd/PortalConfig_6.1.0.xsd");

    /**
     * Define value of FAILED status of Request object.
     */
    private static final String REQUESTSTATUS_FAILED = "FAILED";

    /**
     * Psuedo name to identify generated XML access script
     */
	private static final String SCRIPTNAME_DOM_GENERATED = "--inline generated XML access based upon DOM structure--";

    /**
     * Initialize JAXB Context.
     */
    private static JAXBContext JAXBCONTEXT;

    /**
     * Store container.
     */
    private final WpContainer container;

    /**
     * Initialize executor for specified container.
     * 
     * @param container
     *            The container to execute the XML Access command.
     */
    public XmlAccessExecutor(final WpContainer container) {
        super();

        // Preconditions
        Validate.notNull(container);

        // Store values
        this.container = container;
    }

    /**
     * Execute XML access
     * 
     * @param context
     *            The ExecutionContext in which the command is executed
     * @param domObject
     *            DOM of XML access request to execute
     * @return result of the command
     * @throws XmlAccessExecutionException
     */
    public Request executeXmlAccess(final ExecutionContext context, final Request domObject) throws XmlAccessExecutionException {
        return this.executeXmlAccess(null, context, domObject);
    }

    /**
     * Execute XML access
     * 
     * @param session
     *            The OverthereConnection used to execute the remote command
     * @param context
     *            The ExecutionContext in which the command is executed
     * @param domObject
     *            DOM of XML access request to execute
     * @return result of the command
     * @throws XmlAccessExecutionException
     */
    public Request executeXmlAccess(final OverthereConnection session, final ExecutionContext context, final Request domObject) throws XmlAccessExecutionException {

        // Validate parameters
        if (context == null) {
            throw new IllegalArgumentException("Parameter \"context\" cannot be NULL");
        }
        if (domObject == null) {
            throw new IllegalArgumentException("Parameter \"domObject\" cannot be NULL");
        }

        final OverthereConnection localConnection = LocalConnection.getLocalConnection();
        OverthereFile temporaryFile = null;
        try {
            // Create temporary file
            temporaryFile = localConnection.getTempFile("xmlaccess-request-", ".xml");

            // Marshal data
            final Marshaller marshaller = XmlAccessExecutor.JAXBCONTEXT.createMarshaller();

            // Setup marshaller
            marshaller.setProperty("jaxb.noNamespaceSchemaLocation", XmlAccessExecutor.XSD_FILENAME);

            // Perform marshal
            marshaller.marshal(domObject, temporaryFile.getOutputStream());

            // Now call command with new file
            return this.executeCmd(session, context, temporaryFile, XmlAccessExecutor.SCRIPTNAME_DOM_GENERATED);

        } catch (final JAXBException e) {
            return null;
        } finally {
            if ((temporaryFile != null) && temporaryFile.exists()) {
                temporaryFile.delete();
            }
        }

    }

    public Request executeXmlAccess(final ExecutionContext context, final OverthereFile xmlaccessRequestFile, final String scriptTemplateId) throws XmlAccessExecutionException {
        return this.executeXmlAccess(null, context, xmlaccessRequestFile, scriptTemplateId);
    }

    public Request executeXmlAccess(final OverthereConnection session, final ExecutionContext context, final OverthereFile xmlaccessRequestFile, final String scriptTemplateId) throws XmlAccessExecutionException {

        // Precondition
        if (context == null) {
            throw new IllegalArgumentException("Parameter \"context\" cannot be NULL");
        }
        if (xmlaccessRequestFile == null) {
            throw new IllegalArgumentException("Parameter \"xmlaccessRequestFile\" cannot be NULL");
        }

        // Ensure file exists
        if (xmlaccessRequestFile.exists()) {

            // Validate if XML is valid
            boolean xmlvalid;

            try {
                final JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class.getPackage().getName());
                final Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();

                final Schema schema = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI).newSchema(XmlAccessExecutor.XSD_LOCATION);
                unmarshaller.setSchema(schema);

                // Attempt to unmarshal, not interested in result
                unmarshaller.unmarshal(xmlaccessRequestFile.getInputStream());

                // Update state
                xmlvalid = true;
            } catch (final JAXBException je) {
                // Log error
                XmlAccessExecutor.LOGGER.warn("Invalid XML Access Script found! Execution aborted...", je);

                // Log output
                if ( XmlAccessExecutor.LOGGER.isDebugEnabled() ) {
                	try {
	                	final InputStream stream = xmlaccessRequestFile.getInputStream();
	                	try {
	                		final String generatedXmlAccess = IOUtils.toString(stream);
	                		XmlAccessExecutor.LOGGER.debug("The following XML access content is not valid: \n {}", generatedXmlAccess);
	                	} finally {
	                		IOUtils.closeQuietly(stream);
	                	}
                	} catch(IOException ioe) {
                		XmlAccessExecutor.LOGGER.debug("Unable to generate XML Access snippet due to an IOException", ioe);
                	}
                }
                
                xmlvalid = false;
            } catch (final SAXException se) {
                // Log error
                XmlAccessExecutor.LOGGER.warn("Invalid XML Access Script found! Execution aborted...", se);

                // Log output
                if ( XmlAccessExecutor.LOGGER.isDebugEnabled() ) {
                	try {
	                	final InputStream stream = xmlaccessRequestFile.getInputStream();
	                	try {
	                		final String generatedXmlAccess = IOUtils.toString(stream);
	                		XmlAccessExecutor.LOGGER.debug("The following XML access content is not valid: \n {}", generatedXmlAccess);
	                	} finally {
	                		IOUtils.closeQuietly(stream);
	                	}
                	} catch(IOException ioe) {
                		XmlAccessExecutor.LOGGER.debug("Unable to generate XML Access snippet due to an IOException", ioe);
                	}
                }

                xmlvalid = false;
            }

            // Ensure xml is valid
            if (xmlvalid) {
                return this.executeCmd(session, context, xmlaccessRequestFile, scriptTemplateId);
            }
        }

        // Cannot execute
        return null;

    }

    /**
     * Determine if request was successfully executed.
     * 
     * @param xmlAccessResponse
     *            Response object
     * @return TRUE if all actions are successfully executed (status = OK), FALSE otherwise. Status WARNING and ERROR results in failed execution.
     */
    public boolean isSuccessfullyExecuted(final Request xmlAccessResponse) {

        boolean successfull;

        // Ensure response is valid
        if (xmlAccessResponse != null) {
            // Valid response, test response for errors

            // If no error is found, then state is successful
            successfull = true;

            // Loop all status elements
            for (final StatusInfo statusInfo : xmlAccessResponse.getStatus()) {
                // Check if status is OK
                if (XmlAccessExecutor.REQUESTSTATUS_FAILED.equalsIgnoreCase(statusInfo.getResult())) {
                    // Not OK, not successful
                    successfull = false;
                    // Break loop, status is defined
                    break;
                }
            }
        } else {
            // NULL is never successful
            successfull = false;
        }

        return successfull;
    }

    /**
     * Execute the command using the specified session and context.
     * 
     * @param session
     *            The OverthereConnection used to execute the remote command.
     * @param context
     *            The ExecutionContext in which the command is executed.
     * @param xmlaccessRequestFile
     *            The XML Access script with the commands to execute.
     * @param scriptTemplateId 
     * @return Result of the execution
     * @throws XmlAccessExecutionException
     */
    protected Request executeCmd(OverthereConnection session, final ExecutionContext context, final OverthereFile xmlaccessRequestFile, final String scriptTemplateId) throws XmlAccessExecutionException {

        // Define return value
        Request result;

    	if ( XmlAccessExecutor.SCRIPTLOGGER.isTraceEnabled() ) {
			InputStream inputStream = xmlaccessRequestFile.getInputStream();
    		try {
        		final String templateContent = IOUtils.toString(inputStream);
        		ScriptUtils.dumpScript(scriptTemplateId, templateContent, XmlAccessExecutor.SCRIPTLOGGER);
    		} catch(IOException ioe) {
    			XmlAccessExecutor.LOGGER.warn(String.format("Unable to read XML access from file [%s]", xmlaccessRequestFile.getName()),ioe);
    		} finally {
    			IOUtils.closeQuietly(inputStream);
    		}
    	}        
        
        // Create session if non supplied
        if (session == null) {

            // Create new connection
            session = this.container.getRuntimeContainer().getHost().getConnection();
            try {
                // Execute command using daemon
                result = this.__performExecute(session, context, xmlaccessRequestFile);
            } finally {
                session.close();
            }
        } else {
            // Execute command
            result = this.__performExecute(session, context, xmlaccessRequestFile);
        }

        // Execute command using daemon
        return result;
    }

    private Request __performExecute(final OverthereConnection targetHostConnection, final ExecutionContext ctx, final OverthereFile xmlaccessRequestFile) throws XmlAccessExecutionException {

        // Get connection with target host
        final WpRuntimeContainer runtimeContainer = this.container.getRuntimeContainer();

        // Log
        this.log(ctx, "Copy XML Access to target.");

        // Copy to target server
        final OverthereFile executableScript = targetHostConnection.getTempFile("execute_xmlaccess", ".xml");
        xmlaccessRequestFile.copyTo(executableScript);

        // Log
        this.log(ctx, MessageFormat.format("Finished coping XML Access to target. File is {0}", executableScript.getPath()));

        // Define temporary output file
        final OverthereFile targetOutput = targetHostConnection.getTempFile("execute_xmlaccess", "_out.xml");

        // Compose command
        final StringBuilder xmlAccessCmd = new StringBuilder();
        xmlAccessCmd.append(runtimeContainer.getWpHome());
        xmlAccessCmd.append(targetHostConnection.getHostOperatingSystem().getFileSeparator());
        xmlAccessCmd.append("bin");
        xmlAccessCmd.append(targetHostConnection.getHostOperatingSystem().getFileSeparator());
        xmlAccessCmd.append("xmlaccess");
        xmlAccessCmd.append(targetHostConnection.getHostOperatingSystem().getScriptExtension());
        final CmdLine commandLine = CmdLine.build(xmlAccessCmd.toString());
        commandLine.addArgument("-url");
        commandLine.addArgument(this.container.getWpConfigUrl());

        commandLine.addArgument("-user");
        commandLine.addArgument(runtimeContainer.getWpAdminUsername());

        commandLine.addArgument("-password");
        commandLine.addPassword(runtimeContainer.getWpAdminPassword());

        commandLine.addArgument("-in");
        commandLine.addArgument(executableScript.getPath());

        commandLine.addArgument("-out");
        commandLine.addArgument(targetOutput.getPath());

        // Log
        this.log(ctx, MessageFormat.format("Start XML Access script with command : {0}", commandLine.toCommandLine(targetHostConnection.getHostOperatingSystem(), true)));

        // Do execute
        final int executionResult = targetHostConnection.execute(new DefaultProcessOutputHandler(ctx), commandLine);

        if ( (executionResult == 0) || (executionResult == 1) ) {
            // Returned successful

        	final InputStream input = targetOutput.getInputStream();
            try {
            	final String xmlAccessOutput = IOUtils.toString(input);
            	ctx.logOutput(String.format("XMLAccess response:\n%s",xmlAccessOutput));
                XmlAccessExecutor.LOGGER.debug("Remote response: {}", xmlAccessOutput);
            } catch (final IOException ioe) {
                XmlAccessExecutor.LOGGER.error("Unable to catch output response for logging", ioe);
                ctx.logError("Unable to retrieve XML Access response due to an exception", ioe);
            } finally {
                IOUtils.closeQuietly(input);                	
            }

            // Dump response if in debug
            if (XmlAccessExecutor.LOGGER.isDebugEnabled()) {
            }

            // Define return value
            Request returnValue = null;

            // Parse XML
            try {
                final JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class.getPackage().getName());
                final Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
                final Object unmarshalledObject = unmarshaller.unmarshal(targetOutput.getInputStream());

                // Validate result
                if (unmarshalledObject instanceof Request) {
                    returnValue = (Request) unmarshalledObject;
                }

                this.log(ctx, "Finished executing XML Access script");

            } catch (final JAXBException e) {
                // Wrap exception
                new XmlAccessExecutionException("Unable to process result due to JAXException. XML Access script already executed!", e);
            }

            return returnValue;
        } else {
            // Not successful, return fail
            this.logError(ctx, MessageFormat.format("Unable to execute XML Access script due to error code {0}", executionResult));
            return null;
        }

    }

    /**
     * Convenient method to log to both context and logfile.
     * 
     * @param ctx
     *            Context used to log.
     * @param message
     *            Message to log.
     */
    private void log(final ExecutionContext ctx, final String message) {
        ctx.logOutput(message);
        XmlAccessExecutor.LOGGER.debug(message);
    }

    /**
     * Convenient method to log errors to both context and logfile.
     * 
     * @param ctx
     *            Context used to log.
     * @param message
     *            Message to log.
     */
    private void logError(final ExecutionContext ctx, final String message) {
        ctx.logError(message);
        XmlAccessExecutor.LOGGER.error(message);
    }

    static {
        // Initialize context
        try {
            XmlAccessExecutor.JAXBCONTEXT = JAXBContext.newInstance(ObjectFactory.class.getPackage().getName());
        } catch (final JAXBException je) {
            throw new RuntimeException("Unable to initialize JAXB context for XmlAccess", je);
        }
    }
}
