package com.crossingchannels.portal.websphere.specification.jee5;

import java.io.InputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.xml.bind.JAXBException;

import org.apache.commons.lang.Validate;

import com.crossingchannels.portal.websphere.specification.jee5.dom.ApplicationType;
import com.crossingchannels.portal.websphere.specification.jee5.dom.ModuleType;
import com.crossingchannels.portal.websphere.specification.jee5.dom.ObjectFactory;
import com.crossingchannels.portal.websphere.specification.jee5.dom.WebType;
import com.crossingchannels.portal.websphere.util.ChangeNamespaceXmlFilter;
import com.crossingchannels.portal.websphere.util.UnexpectedXmlElementException;
import com.crossingchannels.portal.websphere.util.XmlUtil;

/**
 * Represents an JEE version 5 compatible EAR descriptor.
 * 
 * @author FWiegerinck
 */
public class EarDescriptor {

    private static final Set<String> COMPATIBLE_NAMESPACES = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList("http://java.sun.com/xml/ns/javaee", "")));

    private static final String TARGET_NAMESPACE = "http://java.sun.com/xml/ns/javaee";

    // /**
    // * Create a new EarDescriptor object using the supplied XML file.
    // *
    // * @param earXml
    // * Reference to the XML file.
    // * @return Instance of EarDescriptor.
    // * @throws FileNotFoundException
    // * Thrown if the XML file could not be found or read.
    // * @throws IllegalDescriptorFoundException
    // */
    // public static final EarDescriptor fromFile(final File earXml) throws FileNotFoundException, IllegalDescriptorFoundException {
    // Validate.notNull(earXml);
    //
    // // Ensure file is valid
    // if (!earXml.exists() || !earXml.canRead()) {
    // throw new FileNotFoundException(String.format("Unable to read file [%s]", earXml.getAbsolutePath()));
    // }
    //
    // final InputStream inputStream = new FileInputStream(earXml);
    // }

    /**
     * Create a new EarDescriptor object using the supplied XML file.
     * 
     * @param earXmlInputStream
     *            InputStream containing the XML file.
     * @return Instance of EarDescriptor.
     * @throws IllegalDescriptorFoundException
     *             Thrown if the descriptor is not valid.
     */
    public static final EarDescriptor fromInputStream(final InputStream earXmlInputStream) throws IllegalDescriptorFoundException {
        Validate.notNull(earXmlInputStream);
        // Load DOM
        final ApplicationType domApplication;
        try {
            domApplication = XmlUtil.loadDom(earXmlInputStream, ObjectFactory.class.getPackage(), ApplicationType.class, new ChangeNamespaceXmlFilter(EarDescriptor.TARGET_NAMESPACE,
                    EarDescriptor.COMPATIBLE_NAMESPACES));
        } catch (final JAXBException e) {
            throw new IllegalDescriptorFoundException("EAR descriptor contains invalid XML", e);
        } catch (final UnexpectedXmlElementException e) {
            throw new IllegalDescriptorFoundException("EAR descriptor contains illegal XML root element", e);
        }

        // Create return object
        final EarDescriptor earDescriptor = new EarDescriptor();
        if (!domApplication.getDisplayName().isEmpty()) {
            earDescriptor.setDisplayName(domApplication.getDisplayName().get(0).getValue());
        }
        for (final ModuleType moduleType : domApplication.getModule()) {
            if (moduleType.getWeb() != null) {
                final WebType webType = moduleType.getWeb();
                earDescriptor.addWebArchive(webType.getWebUri().getValue(), webType.getContextRoot().getValue());
            }
        }

        return earDescriptor;
    }

    private String displayName;
    private final Map<String, String> webModules = new HashMap<String, String>();

    /**
     * Add a new web archive to this descriptor.
     * 
     * @param webArchiveURI
     *            URI of the web archive to add.
     * @param contextRoot
     *            Context root of the archive.
     */
    public void addWebArchive(final String webArchiveURI, final String contextRoot) {
        this.webModules.put(webArchiveURI, contextRoot);
    }

    /**
     * @return the displayName
     */
    public String getDisplayName() {
        return this.displayName;
    }

    /**
     * Get the context root of the requested web archive.
     * 
     * @param webArchiveURI
     *            URI of the web archive requested.
     * @return The context root or NULL if non set.
     */
    public String getWebArchiveContextRoot(final String webArchiveURI) {
        return this.webModules.get(webArchiveURI);
    }

    /**
     * Get a set of the URIs of all web archive registered in this descriptor.
     * 
     * @return Unmodifable set of URIs.
     */
    public Set<String> getWebArchiveURIs() {
        return Collections.unmodifiableSet(this.webModules.keySet());
    }

    /**
     * @param displayName
     *            the displayName to set
     */
    public void setDisplayName(final String displayName) {
        this.displayName = displayName;
    }

}
