package com.crossingchannels.portal.websphere.util;

import java.io.InputStream;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.sax.SAXSource;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.lang.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLFilter;
import org.xml.sax.XMLReader;

/**
 * Utility class to XML operations.
 * 
 * @author FWiegerinck
 */
public class XmlUtil {

    /**
     * Define logger for this class.
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(XmlUtil.class);

    /**
     * Load XML from supplied input stream and create DOM objects from specified package.
     * 
     * @param inputStream
     *            The input stream containing the XML. Stream will not be closed at the end of the parsing
     * @param domPackage
     *            The package containing DOM objects.
     * @param expectedDomType
     *            The expected DOM object to return, or NULL if no validation is required.
     * @return The expected DOM object.
     * @throws JAXBException
     *             Thrown if an error occurs during parsing of the XML.
     * @throws UnexpectedXmlElementException
     *             Thrown if an unexpected root XML element has been found.
     */
    @SuppressWarnings("unchecked")
    public static <T> T loadDom(final InputStream inputStream, final Package domPackage, final Class<T> expectedDomType, final XMLFilter xmlFilter) throws JAXBException, UnexpectedXmlElementException {
        // Preconditions
        Validate.notNull(inputStream);
        Validate.notNull(domPackage);

        // Start processing stream
        XmlUtil.LOGGER.trace("Start loading XML from stream. DOM objects are located inside package [{}]. Expected DOM object to return is [{}]", domPackage, expectedDomType);

        final JAXBContext jc = JAXBContext.newInstance(domPackage.getName());
        final Unmarshaller unmarshaller = jc.createUnmarshaller();

        final InputSource inputSource = new InputSource(inputStream);
        final SAXParserFactory factory = SAXParserFactory.newInstance();
        factory.setNamespaceAware(true);
        XMLReader reader;
        try {
            reader = factory.newSAXParser().getXMLReader();
        } catch (final SAXException e) {
            throw new JAXBException("Unable to initialize SAX parser", e);
        } catch (final ParserConfigurationException e) {
            throw new JAXBException("Unable to initialize SAX parser due to parser configuration error", e);
        }

        reader.setContentHandler(unmarshaller.getUnmarshallerHandler());

        Object domObj;
        if (xmlFilter != null) {
            xmlFilter.setParent(reader);

            domObj = unmarshaller.unmarshal(new SAXSource(xmlFilter, inputSource));
        } else {
            domObj = unmarshaller.unmarshal(new SAXSource(reader, inputSource));
        }

        // Get value if domObj is a wrapper
        if (domObj instanceof JAXBElement<?>) {
            domObj = ((JAXBElement<?>) domObj).getValue();
        }

        if ((expectedDomType != null) && (!expectedDomType.isInstance(domObj))) {
            throw new UnexpectedXmlElementException(expectedDomType, domObj);
        }

        // Expected type, return instance
        return (T) domObj;

    }
    
    /**
     * Load XML from supplied InputStream and search for the expression.
     * 
     * @param expression
     * 		The expression to evaluate
     * @param stream
     * 		The InputStream containing the XML
     * @return
     * 		The value of the evaluated expression
     * @throws XPathExpressionException
     * 		Thrown when an error occurs evaluating the expression
     */
    public static String getValue(final String expression, final InputStream stream) throws XPathExpressionException {
    	// Create XPathFactory
    	XPathFactory factory = XPathFactory.newInstance();
    	// Create XPath
    	XPath xPath = factory.newXPath();
 
    	// Evaluate the expression and return the value found
    	return xPath.evaluate(expression, new InputSource(stream));
    }

    /**
     * Hide constructor for utility class.
     */
    private XmlUtil() {
        // Do nothing
    }
}
