package com.crossingchannels.portal.websphere.template;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.udm.Deployed;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

/**
 * Base implementation of a freemarker template parser. This object is thread-safe.
 * 
 * @author FWiegerinck
 */
public class BaseFreemarkerTemplateParser implements TemplateParser {

    /**
     * Define constant with name of deployed CI which can be used with the freemarker template.
     */
    public static final String VAR_DEPLOYED_CI = "deployed";

    /**
     * Define logger for this class.
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(BaseFreemarkerTemplateParser.class);

    /**
     * Store the freemarker configuration.
     */
    private final Configuration freemarkerConfiguration;

    /**
     * Initialize new base freemarker template parser.
     * 
     * @param freemarkerConfiguration
     *            The configuration used to locate and parse templates.
     */
    protected BaseFreemarkerTemplateParser(final Configuration freemarkerConfiguration) {
        super();

        // Store configuration.
        this.freemarkerConfiguration = freemarkerConfiguration;
    }
    
    protected BaseFreemarkerTemplateParser() {
    	this.freemarkerConfiguration = null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void parse(final String templateName, final Map<String, ?> context, final OutputStream out) throws TemplateParseException {

        // Preconditions
        Validate.notEmpty(templateName);
        Validate.notNull(context);
        Validate.notNull(out);

        try {
            // Get template
            final Template template = this.freemarkerConfiguration.getTemplate(templateName);

            // Do parse
            template.process(context, new OutputStreamWriter(out));
        } catch (final IOException ioe) {
            // Log error
            BaseFreemarkerTemplateParser.LOGGER.error(String.format("Unable to read/find template [%s]", templateName), ioe);

            // Wrap exception
            throw new TemplateParseException(templateName, ioe);
        } catch (final TemplateException te) {
            // Log error
            BaseFreemarkerTemplateParser.LOGGER.error(String.format("Unable to parse template [%s]", templateName), te);

            // Wrap exception
            throw new TemplateParseException(templateName, te);
        }

    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void parse(final String templateName, final Deployed<?, ?> deployedCi, final Map<String, ?> context, final OutputStream out) throws TemplateParseException {

        // Preconditions
        Validate.notNull(templateName);
        Validate.notNull(deployedCi);
        Validate.notNull(context);
        Validate.notNull(out);

        // Create new map with context
        final Map<String, Object> mergedContext = new HashMap<String, Object>(context);
        mergedContext.put(BaseFreemarkerTemplateParser.VAR_DEPLOYED_CI, deployedCi);

        // Do initiate evaluation of template
        this.parse(templateName, mergedContext, out);
    }
}
