package com.xebialabs.deployit.plugin.cloud.step;


import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import com.google.common.base.Predicate;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.cloud.ci.InstantiatableCloudTemplate;
import com.xebialabs.deployit.plugin.cloud.util.ContextHelper;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Collections2.filter;

/**
 * Registers created CIs in repository. It also puts all Container CIs in context attribute
 * {@code ContextAttribute.REGISTERED_CONTAINERS} and Host CIs in {@code ContextAttribute.REGISTERED_HOSTS}. This step
 * depends on context attributes {@code ContextAttribute.CREATED_INSTANCES} and {@code ContextAttribute.USED_TEMPLATES},
 * which should contain created instances and used templates in the proper order. This step always should be executed
 * after some step, which actually triggers instantiation on cloud platform and writes appropriate objects to the
 * context.
 */
@SuppressWarnings("serial")
public class RegisterInstancesStep implements Step {

    /**
     * Directory which should contain created hosts
     */
    private final String hostsPath;

    public RegisterInstancesStep(String hostsPath) {
        this.hostsPath = hostsPath;
    }

    @Override
    public int getOrder() {
        return DEFAULT_ORDER + 30;
    }

    @Override
    public String getDescription() {
        return "Register created instances in repository";
    }

    @Override
    @SuppressWarnings("unchecked")
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {

        List<String> instances;
        List<InstantiatableCloudTemplate> templates;

        try {
            instances = checkNotNull((List<String>) ctx.getAttribute(ContextAttribute.CREATED_INSTANCES.name()));
            templates = checkNotNull((List<InstantiatableCloudTemplate>) ctx.getAttribute(ContextAttribute.USED_TEMPLATES.name()));
        } catch (NullPointerException e) {
            ctx.logError("Can not find required context attribute", e);
            return StepExitCode.FAIL;
        }

        for(int i=0; i < instances.size(); i++) {

            InstantiatableCloudTemplate instanceTpl = templates.get(i);
            String cloudId = instances.get(i);
            List<ConfigurationItem> cis = instanceTpl.produceInstanceCIs(cloudId, ctx.getRepository(), hostsPath);

            for (ConfigurationItem ci : cis) {
                ctx.logOutput("Configuration item " + ci.getId() + " has been created");
            }

            ctx.getRepository().create(cis.toArray(new ConfigurationItem[cis.size()]));

            Collection<ConfigurationItem> hosts = filter(cis, new Predicate<ConfigurationItem>() {
                @Override
                public boolean apply(final ConfigurationItem ci) {
                    return ci.getType().getDescriptor().isAssignableTo(Type.valueOf("overthere.Host"));
                }
            });

            Collection<ConfigurationItem> containers = filter(cis, new Predicate<ConfigurationItem>() {
                @Override
                public boolean apply(final ConfigurationItem ci) {
                    return ci.getType().getDescriptor().isAssignableTo(Type.valueOf("udm.Container"));
                }
            });

            for (ConfigurationItem host : hosts) {
                ctx.logOutput("Host " + host.getProperty("address") + " with underlying middleware is created");
            }

            ContextHelper.wrapped(ctx).safeSet(ContextAttribute.REGISTERED_CONTAINERS.name(), new HashSet<ConfigurationItem>(), containers);
            ContextHelper.wrapped(ctx).safeSet(ContextAttribute.REGISTERED_HOSTS.name(), new HashSet<ConfigurationItem>(), hosts);
        }


        return StepExitCode.SUCCESS;
    }

}
