package com.xebialabs.deployit.plugin.cloud.ci;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeoutException;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.cloud.step.RegisterInstancesStep;
import com.xebialabs.deployit.plugin.cloud.util.HostFactory;
import com.xebialabs.deployit.plugin.cloud.util.MarkerChecker;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.ConnectionOptions;

import static com.google.common.collect.Lists.newArrayList;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "Base class for host instance templates, all independently instantiatable host instance templates must extend it")
public abstract class InstantiatableHostTemplate extends BaseHostTemplate implements InstantiatableCloudTemplate {
    private MarkerChecker markerChecker = new MarkerChecker();

    /* for testing purposes only */
    void setMarkerChecker(MarkerChecker markerChecker) {
        this.markerChecker = markerChecker;
    }

    public List<? extends Step> produceValidationSteps() {
        return validateDescriptor();
    }

    @ControlTask(label = "Instantiate", description = "Create instance from template", parameterType = "cloud.HostParameters")
    public List<? extends Step> instantiate(HostParameters params) {
        ArrayList<Step> steps = newArrayList(produceCreateSteps(params.getInstanceName(), 0));
        steps.add(new RegisterInstancesStep(params.getHostsLocation()));
        return steps;
    }

    @Override
    public List<ConfigurationItem> produceInstanceCIs(String cloudId, Repository repository, String hostsPath) throws TimeoutException {
        String instancePublicAddress = getInstanceIpAddress(cloudId);
        return produceInstanceCIs(cloudId, repository, hostsPath, instancePublicAddress);
    }

    /**
     * Should return instance public IP address *and* wait until instance is considered running according to hypervisor
     */
    public abstract String getInstanceIpAddress(String cloudId) throws TimeoutException;

    @Override
    public void waitForAvailability(ExecutionContext ctx, String cloudId) throws TimeoutException {
        /* always wait for availability of ip address */
        String instancePublicAddress = getInstanceIpAddress(cloudId);

        if (getMarkerPath() != null) {
            Host host = HostFactory.fromTemplate(this);
            host.setProperty("address", instancePublicAddress);

            ConnectionOptions connectionOptions = new Host.ConnectionOptionsBuilder().getConnectionOptions(host);

            markerChecker.waitForMarkerFileAppears(
                (String) host.getProperty("protocol"),
                connectionOptions,
                getMarkerPath(),
                getBootTimeout(),
                getRetryDelay()
                );
        }
    }

}
