package com.xebialabs.deployit.plugin.cloud.vsphere.ci;

import java.util.List;
import java.util.concurrent.TimeoutException;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.cloud.ci.InstantiatableHostTemplate;
import com.xebialabs.deployit.plugin.cloud.vsphere.access.VsphereAdapter;
import com.xebialabs.deployit.plugin.cloud.vsphere.steps.CreateVSphereInstanceStep;
import com.xebialabs.deployit.plugin.cloud.vsphere.steps.DestroyInstanceStep;
import com.xebialabs.deployit.plugin.cloud.vsphere.steps.WaitForVsphereIpsStep;
import com.xebialabs.deployit.plugin.cloud.vsphere.steps.WaitForVsphereTasksStep;

import static java.util.Arrays.asList;

@Metadata(description = "vSphere instance template", root = Metadata.ConfigurationItemRoot.CONFIGURATION)
public class HostTemplate extends InstantiatableHostTemplate {

    @Property(required = true, label = "vCenter credentials")
    private Credentials credentials;

    @Property(required = true, label = "Datacenter", description = "Name of the datacenter.")
    private String datacenter;

    @Property(required = true, label = "Template path", description = "Path to the template relative to the datacenter. The datacenter name should not be included in the path.")
    private String templatePath;

    @Property(required = true, label = "Host", description = "Name of the host or cluster on which to run the virtual machine.")
    private String host;

    @Property(required = false, label = "Destination path", description = "Path to the folder where the new virtual machine should be created, relative to the datacenter. Leave this field empty if you want to create the virtual machine directly under the datacenter.")
    private String destinationPath;

    @Property(required = false, label = "Resource pool", description = "Name of the resource pool to be used for the virtual machine.")
    private String resourcePool;

    @Property(required = false, label = "Datastore", description = "Name of the datastore to be used for the virtual machine.")
    private String datastore;

    @Property(required = false, label = "Memory (MB)", description = "Amount of memory to be allocated for the virtual machine in megabytes.")
    private int memory;

    @Property(required = false, label = "CPUs", description = "Amount of CPUs to be available for the virtual machine.")
    private int cpus;

    @Property(required = false, label = "Customization specification", description = "Name of the existing customization specification.")
    private String customization;



    public Credentials getCredentials() {
        return credentials;
    }

    public String getDatacenter() {
        return datacenter;
    }

    public String getTemplatePath() {
        return templatePath;
    }

    public String getResourcePool() {
        return resourcePool;
    }

    public int getMemory() {
        return memory;
    }

    public int getCpus() {
        return cpus;
    }

    public void setCredentials(final Credentials credentials) {
        this.credentials = credentials;
    }

    public void setDatacenter(final String datacenter) {
        this.datacenter = datacenter;
    }

    public void setTemplatePath(final String templatePath) {
        this.templatePath = templatePath;
    }

    public void setResourcePool(final String resourcePool) {
        this.resourcePool = resourcePool;
    }

    public void setMemory(final int memory) {
        this.memory = memory;
    }

    public void setCpus(final int cpus) {
        this.cpus = cpus;
    }

    public String getHost() {
        return host;
    }

    public void setHost(final String host) {
        this.host = host;
    }

    public String getDestinationPath() {
        return destinationPath;
    }

    public void setDestinationPath(final String destinationPath) {
        this.destinationPath = destinationPath;
    }

    public String getDatastore() {
        return datastore;
    }

    public void setDatastore(final String datastore) {
        this.datastore = datastore;
    }

    public String getCustomization() {
        return customization;
    }

    public void setCustomization(final String customization) {
        this.customization = customization;
    }

    @Override
    public List<? extends Step> produceCreateSteps(String envName, int instanceSeq) {

        String instanceLabel = envName + " (" + getName() + ") #" + instanceSeq;

        VsphereAdapter adapter = new VsphereAdapter(getCredentials());

        return asList(
                new CreateVSphereInstanceStep(this, instanceLabel, new VsphereAdapter(getCredentials())),
                new WaitForVsphereTasksStep(this, adapter),
                new WaitForVsphereIpsStep(this, adapter)
        );
    }

    @Override
    public List<? extends Step> produceDestroySteps(ConfigurationItem instance) {
        VsphereAdapter adapter = new VsphereAdapter(getCredentials());
        return asList(new DestroyInstanceStep(instance, adapter), new WaitForVsphereTasksStep(this, adapter));
    }

    @Override
    public String getInstanceIpAddress(String cloudId) throws TimeoutException {
        return new VsphereAdapter(getCredentials()).getIpAddress(cloudId);
    }
}
