package com.xebialabs.deployit.plugin.cloud.vsphere.steps;

import java.util.ArrayList;
import java.util.NoSuchElementException;

import com.google.common.base.Predicate;
import com.google.common.base.Throwables;
import com.vmware.vim25.VirtualMachineCloneSpec;
import com.vmware.vim25.VirtualMachineConfigSpec;
import com.vmware.vim25.VirtualMachineRelocateSpec;
import com.vmware.vim25.mo.ComputeResource;
import com.vmware.vim25.mo.CustomizationSpecManager;
import com.vmware.vim25.mo.Folder;
import com.vmware.vim25.mo.InventoryNavigator;
import com.vmware.vim25.mo.ManagedEntity;
import com.vmware.vim25.mo.ResourcePool;
import com.vmware.vim25.mo.Task;
import com.vmware.vim25.mo.VirtualMachine;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.cloud.ci.InstantiatableCloudTemplate;
import com.xebialabs.deployit.plugin.cloud.vsphere.access.VsphereAdapter;
import com.xebialabs.deployit.plugin.cloud.vsphere.ci.HostTemplate;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Iterables.getOnlyElement;
import static com.xebialabs.deployit.plugin.cloud.step.ContextAttribute.CREATED_INSTANCES;
import static com.xebialabs.deployit.plugin.cloud.step.ContextAttribute.STARTED_TASKS;
import static com.xebialabs.deployit.plugin.cloud.step.ContextAttribute.USED_TEMPLATES;
import static com.xebialabs.deployit.plugin.cloud.util.ContextHelper.wrapped;
import static com.xebialabs.deployit.plugin.cloud.vsphere.access.VsphereUtils.vmPath;
import static java.util.Arrays.asList;

public class CreateVSphereInstanceStep implements Step {

    private HostTemplate template;

    private String instanceName;

    private VsphereAdapter adapter;

    public CreateVSphereInstanceStep(final HostTemplate template, final String name, final VsphereAdapter adapter) {
        this.template = template;
        this.instanceName = name;
        this.adapter = adapter;
    }

    @Override
    public int getOrder() {
        return DEFAULT_ORDER;
    }

    @Override
    public String getDescription() {
        return "Deploy a virtual machine from " + template.getTemplatePath();
    }

    @Override
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {
        try {

            ResourcePool pool;
            ComputeResource computeResource = adapter.getHost(template.getDatacenter(), template.getHost());

            if (isNullOrEmpty(template.getResourcePool())) {
                ctx.logOutput("Using the whole host as a resource pool.");
                if (computeResource == null) {
                    ctx.logError(String.format("Could not locate host '%s' in datacenter '%s'", template.getDatacenter(), template.getHost()));
                    return StepExitCode.FAIL;
                }
                pool = computeResource.getResourcePool();
            } else {
                ctx.logOutput("Using resource pool " + template.getResourcePool());
                InventoryNavigator parent = new InventoryNavigator(adapter.getHost(template.getDatacenter(), template.getHost()));

                try {
                    pool = filterManagedEntities(parent.searchManagedEntities("ResourcePool"), template.getResourcePool());
                } catch (NoSuchElementException e) {
                    ctx.logError("Can not find resource pool " + template.getResourcePool(), e);
                    return StepExitCode.FAIL;
                }
            }

            VirtualMachine vmTemplate = adapter.getVm(template.getDatacenter(), template.getTemplatePath());

            // Create
            VirtualMachineCloneSpec spec = new VirtualMachineCloneSpec();
            spec.setPowerOn(true);
            spec.setTemplate(false);

            spec.setLocation(new VirtualMachineRelocateSpec());
            spec.getLocation().setPool(pool.getMOR());

            if (!isNullOrEmpty(template.getDatastore())) {
                try {
                    spec.getLocation().setDatastore(filterManagedEntities(computeResource.getDatastores(), template.getDatastore()).getMOR());
                } catch (NoSuchElementException e) {
                    ctx.logError("Can not find datastore " + template.getDatastore(), e);
                    return StepExitCode.FAIL;
                }
            }

            if(!isNullOrEmpty(template.getCustomization())) {
                CustomizationSpecManager csm = adapter.getSi().getCustomizationSpecManager();
                try {
                    ctx.logOutput("Using customization " + template.getCustomization());
                    spec.setCustomization(csm.getCustomizationSpec(template.getCustomization()).getSpec());
                } catch (com.vmware.vim25.NotFound e) {
                    ctx.logError("Can not find customization " + template.getCustomization(), e);
                    return StepExitCode.FAIL;
                }
            }

            spec.setConfig(new VirtualMachineConfigSpec());
            spec.getConfig().setName(instanceName);

            if (template.getMemory() != 0) {
                ctx.logOutput("Setting memory to " + template.getMemory() + " MB");
                spec.getConfig().setMemoryMB((long) template.getMemory());
            }

            if (template.getCpus() != 0) {
                ctx.logOutput("Setting number of CPUs to " + template.getCpus());
                spec.getConfig().setNumCPUs(template.getCpus());
            }

            Folder destinationFolder = adapter.getVmFolder(template.getDatacenter(), template.getDestinationPath());

            Task task = vmTemplate.cloneVM_Task(destinationFolder, instanceName, spec);

            String expectedVmPath = vmPath(
                    template.getDatacenter(), template.getDestinationPath(), instanceName
            );

            wrapped(ctx).safeSet(USED_TEMPLATES, new ArrayList<InstantiatableCloudTemplate>(), asList((InstantiatableCloudTemplate) template));
            wrapped(ctx).safeSet(CREATED_INSTANCES, new ArrayList<String>(), asList(expectedVmPath));
            wrapped(ctx).safeSet(STARTED_TASKS, new ArrayList<String>(), asList(task.getMOR().getVal()));

        } catch (Exception e) {
            Throwables.propagate(e);
        }

        return StepExitCode.SUCCESS;
    }

    private <T extends ManagedEntity> T filterManagedEntities(final ManagedEntity[] entities, final String name) {
        return (T)getOnlyElement(filter(asList(entities), new Predicate<ManagedEntity>() {
            @Override
            public boolean apply(final ManagedEntity rp) {
                return name.equals(rp.getName());
            }
        }));
    }
}
