import os


def create_loader():
    return Loader()


import com.xebialabs.deployit.plugin.ec2.support.BotoLoader as BotoLoader


class Loader(object):
    def __init__(self):
        self._cache = {}
        self._search_paths = []

    @property
    def search_paths(self):
        return self._search_paths

    def list_available_services(self, type_name):
        return BotoLoader.listAvailableServices()

    def determine_latest_version(self, service_name, type_name):
        return max(self.list_api_versions(service_name, type_name))

    def list_api_versions(self, service_name, type_name):
        return BotoLoader.listApiVersion(service_name, type_name)

    def load_service_model(self, service_name, type_name, api_version=None):
        if api_version is None:
            api_version = self.determine_latest_version(
                service_name, type_name)
        full_path = os.path.join(service_name, api_version, type_name)
        return self.load_data(full_path)

    def load_data(self, name):
        import json

        return json.loads(BotoLoader.loadFile(name))


# Botocore looks for the HOME variable but UNIX init.d service doesn't pass environment variables.
# The script breaks if it doesn't find the variable, thus setting it externally.

if 'HOME' not in os.environ:
    os.environ['HOME'] = os.getenv('xldeploy_home', '/')

from botocore.session import Session as BotocoreSession

from boto3.session import Session


class EC2Helper(object):
    def __init__(self, deployed):
        self.deployed = deployed

        botocore_session = BotocoreSession()
        botocore_session.lazy_register_component('data_loader',
                                                 lambda: create_loader())

        self.session = Session(aws_access_key_id=deployed.container.accesskey,
                               aws_secret_access_key=deployed.container.accessSecret,
                               botocore_session=botocore_session)
        self.ec2 = self.session.resource('ec2', region_name=deployed.region, use_ssl=False)
        self.ec2_client = self.session.client('ec2', region_name=deployed.region, use_ssl=False)
        if hasattr(deployed, 'instanceId') and deployed.instanceId is not None:
            self.instance = self.ec2.Instance(deployed.instanceId)

    def create_instance(self):
        if self.deployed.amiId is None:
            self.deployed.amiId = self.deployed.deployable.name
        instances = self.ec2.create_instances(ImageId=self.deployed.amiId, MinCount=1, MaxCount=1,
                                              InstanceType=self.deployed.instanceType,
                                              SecurityGroups=self.deployed.securityGroup.split(","),
                                              KeyName=self.deployed.keyName)
        if len(instances) > 0:
            self.instance = instances[0]

            print "instance {0} is in {1} state".format(self.instance.instance_id,
                                                    self.instance.state.get("Name"))
            return self.instance
        else:
            raise Exception("Not able to create AWS instance")

    def set_instance_name(self, instance_id, name):
        self.ec2_client.create_tags(DryRun=False, Resources=[instance_id],
                                    Tags=[{'Key': 'Name', 'Value': name}])
        print "updated instance {0} name to {1}".format(instance_id, name)

    def is_instance_running(self, instance_id):
        instance_state_dict = self.ec2_client.describe_instance_status(
            InstanceIds=[
                instance_id,
            ],
            Filters=[{'Name': 'instance-state-name', 'Values': ['running']},
                     {'Name': 'instance-status.reachability', 'Values': ['passed']},
                     {'Name': 'instance-status.status', 'Values': ['ok']},
                     {'Name': 'system-status.status', 'Values': ['ok']}
                     ])
        statuses = instance_state_dict.get("InstanceStatuses")
        if statuses:
            print("instance {0} is in {1} state".format(instance_id, statuses[0]['InstanceState']['Name']))
            self.instance.reload
            return True
        else:
            return False

    def destroy_instance(self):
        ids = [self.deployed.instanceId]
        print "terminate instance {0}".format(self.deployed.instanceId)
        self.ec2.instances.filter(InstanceIds=ids).terminate()

    def public_ip(self):
        return self.instance.public_ip_address

    def public_hostname(self):
        return self.instance.public_dns_name
