from rds.rds_helper import RDSHelper
from ec2.ec2_helper import EC2Helper
from datetime import datetime

class RDSInstanceHelper(RDSHelper):
    def __init__(self, deployed):
        super(RDSInstanceHelper, self).__init__(deployed)
        self.ec2_helper = EC2Helper(deployed)

    def create_instance(self):
        return self.rds_client.create_db_instance(**self.__get_create_instance_params())

    def get_instance(self):
        response = self.rds_client.describe_db_instances(DBInstanceIdentifier=self.deployed.dbInstanceIdentifier)
        return response['DBInstances'][0]

    def destroy_instance(self):
        param_dict = {'DBInstanceIdentifier': self.deployed.dbInstanceIdentifier,
                      'SkipFinalSnapshot': self.deployed.skipFinalSnapshot}
        if not self.deployed.skipFinalSnapshot:
            if self.deployed.finalDBSnapshotIdentifier is None:
                # add a snapshot name with timestamp to avoid collision
                param_dict['FinalDBSnapshotIdentifier'] = self.deployed.dbInstanceIdentifier + \
                    '-final-snapshot-' + datetime.now().strftime('%Y-%m-%dH%H-%M-%S')
            else:
                param_dict['FinalDBSnapshotIdentifier'] = self.deployed.finalDBSnapshotIdentifier
        self.rds_client.delete_db_instance(**param_dict)

    def modify_instance(self, previous_deployed):
        params = self.__get_modify_instance_params(previous_deployed)
        if len(params) > 2:
            return self.rds_client.modify_db_instance(**self.__get_modify_instance_params(previous_deployed))

    def validate_instance_modification(self, previous_deployed):
        if self.deployed.availabilityZone != previous_deployed.availabilityZone:
            raise RuntimeError("Availability zone of RDS instance cannot be updated once created.")
        if self.deployed.masterUsername != previous_deployed.masterUsername:
            raise RuntimeError("Username of RDS instance cannot be updated once created.")
        if self.deployed.characterSetName != previous_deployed.characterSetName:
            raise RuntimeError("Character set name of RDS instance cannot be updated once created.")
        if self.deployed.dbClusterIdentifier != previous_deployed.dbClusterIdentifier:
            raise RuntimeError("DB cluster identifier of RDS instance cannot be updated once created.")
        if self.deployed.storageEncrypted != previous_deployed.storageEncrypted:
            raise RuntimeError("Storage encryption of RDS instance cannot be updated once created.")
        if self.deployed.kmsKeyId != previous_deployed.kmsKeyId:
            raise RuntimeError("Kms key ID of RDS instance cannot be updated once created.")
        if self.deployed.timezone != previous_deployed.timezone:
            raise RuntimeError("Time zone of RDS instance cannot be updated once created.")
        if self.deployed.region != previous_deployed.region:
            raise RuntimeError("Region of RDS instance cannot be updated once created.")
        if self.deployed.dbInstanceIdentifier != previous_deployed.dbInstanceIdentifier:
            raise RuntimeError("DB instance identifier of RDS instance cannot be updated once created.")
        if self.deployed.dbSubnetGroupName != previous_deployed.dbSubnetGroupName:
            raise RuntimeError("DB Subnet Group cannot be updated once created.")
        if self.deployed.dbSecurityGroups != previous_deployed.dbSecurityGroups:
            raise RuntimeError("DB Security Group cannot be updated once created.")

    def __get_create_instance_params(self):
        params_dict = {
            'DBName': self.deployed.dbName,
            'DBInstanceIdentifier': self.deployed.dbInstanceIdentifier,
            'AllocatedStorage': self.deployed.allocatedStorage,
            'DBInstanceClass': self.deployed.dbInstanceClass,
            'Engine': self.deployed.engine,
            'MasterUsername': self.deployed.masterUsername,
            'MasterUserPassword': self.deployed.masterUserPassword,
            'AvailabilityZone': self.deployed.availabilityZone,
            'DBSubnetGroupName': self.deployed.dbSubnetGroupName,
            'PreferredMaintenanceWindow': self.deployed.preferredMaintenanceWindow,
            'DBParameterGroupName': self.deployed.dbParameterGroupName,
            'BackupRetentionPeriod': self.deployed.backupRetentionPeriod,
            'PreferredBackupWindow': self.deployed.preferredBackupWindow,
            'Port': self.deployed.port,
            'MultiAZ': self.deployed.multiAZ,
            'EngineVersion': self.deployed.engineVersion,
            'AutoMinorVersionUpgrade': self.deployed.autoMinorVersionUpgrade,
            'LicenseModel': self.deployed.licenseModel,
            'Iops': self.deployed.iops,
            'OptionGroupName': self.deployed.optionGroupName,
            'CharacterSetName': self.deployed.characterSetName,
            'PubliclyAccessible': self.deployed.publiclyAccessible,
            'DBClusterIdentifier': self.deployed.dbClusterIdentifier,
            'StorageType': self.deployed.storageType,
            'TdeCredentialArn': self.deployed.tdeCredentialArn,
            'TdeCredentialPassword': self.deployed.tdeCredentialPassword,
            'StorageEncrypted': self.deployed.storageEncrypted,
            'KmsKeyId': self.deployed.kmsKeyId,
            'Domain': self.deployed.domain,
            'CopyTagsToSnapshot': self.deployed.copyTagsToSnapshot,
            'MonitoringInterval': self.deployed.monitoringInterval,
            'MonitoringRoleArn': self.deployed.monitoringRoleArn,
            'DomainIAMRoleName': self.deployed.domainIAMRoleName,
            'PromotionTier': self.deployed.promotionTier,
            'Timezone': self.deployed.timezone,
            'EnableIAMDatabaseAuthentication': self.deployed.enableIAMDatabaseAuthentication,
            'DBSecurityGroups': list(self.deployed.dbSecurityGroups),
            'VpcSecurityGroupIds': self.ec2_helper.get_security_group_id_list(self.deployed.vpcSecurityGroupIds),
        }
        not_null_params_dict = {k: v for k, v in params_dict.iteritems() if v is not None}
        return not_null_params_dict

    def __get_modify_instance_params(self, previous_deployed):
        # delta for modification is calculated as not all properties are part of modification call and passing all values as it is takes more time on aws

        modified_params_dict = {'DBInstanceIdentifier': previous_deployed.dbInstanceIdentifier,
                                'ApplyImmediately': self.deployed.applyImmediately}
        if previous_deployed.allocatedStorage != self.deployed.allocatedStorage:
            modified_params_dict['AllocatedStorage'] = self.deployed.allocatedStorage
        if previous_deployed.dbInstanceClass != self.deployed.dbInstanceClass:
            modified_params_dict['DBInstanceClass'] != self.deployed.dbInstanceClass
        if previous_deployed.dbSubnetGroupName != self.deployed.dbSubnetGroupName:
            modified_params_dict['DBSubnetGroupName'] = self.deployed.dbSubnetGroupName
        if previous_deployed.dbSecurityGroups != self.deployed.dbSecurityGroups:
            modified_params_dict['DBSecurityGroups'] = list(self.deployed.dbSecurityGroups)
        if previous_deployed.vpcSecurityGroupIds != self.deployed.vpcSecurityGroupIds:
            modified_params_dict['VpcSecurityGroupIds'] = self.ec2_helper.get_security_group_id_list(self.deployed.vpcSecurityGroupIds)
        if previous_deployed.masterUserPassword != self.deployed.masterUserPassword:
            modified_params_dict['MasterUserPassword'] = self.deployed.masterUserPassword
        if previous_deployed.dbParameterGroupName != self.deployed.dbParameterGroupName:
            modified_params_dict['DBParameterGroupName'] = self.deployed.dbParameterGroupName
        if previous_deployed.backupRetentionPeriod != self.deployed.backupRetentionPeriod:
            modified_params_dict['BackupRetentionPeriod'] = self.deployed.backupRetentionPeriod
        if previous_deployed.preferredBackupWindow != self.deployed.preferredBackupWindow:
            modified_params_dict['PreferredBackupWindow'] = self.deployed.preferredBackupWindow
        if previous_deployed.preferredMaintenanceWindow != self.deployed.preferredMaintenanceWindow:
            modified_params_dict['PreferredMaintenanceWindow'] = self.deployed.preferredMaintenanceWindow
        if previous_deployed.multiAZ != self.deployed.multiAZ:
            modified_params_dict['MultiAZ'] = self.deployed.multiAZ
        if previous_deployed.engineVersion != self.deployed.engineVersion:
            modified_params_dict['EngineVersion'] = self.deployed.engineVersion
        if previous_deployed.autoMinorVersionUpgrade != self.deployed.autoMinorVersionUpgrade:
            modified_params_dict['AutoMinorVersionUpgrade'] = self.deployed.autoMinorVersionUpgrade
        if previous_deployed.licenseModel != self.deployed.licenseModel:
            modified_params_dict['LicenseModel'] = self.deployed.licenseModel
        if previous_deployed.iops != self.deployed.iops:
            modified_params_dict['Iops'] = self.deployed.iops
        if previous_deployed.optionGroupName != self.deployed.optionGroupName:
            modified_params_dict['OptionGroupName'] = self.deployed.optionGroupName
        if previous_deployed.storageType != self.deployed.storageType:
            modified_params_dict['StorageType'] = self.deployed.storageType
        if previous_deployed.tdeCredentialArn != self.deployed.tdeCredentialArn:
            modified_params_dict['TdeCredentialArn'] != self.deployed.tdeCredentialArn
        if previous_deployed.tdeCredentialPassword != self.deployed.tdeCredentialPassword:
            modified_params_dict['TdeCredentialPassword'] = self.deployed.tdeCredentialPassword
        if previous_deployed.domain != self.deployed.domain:
            modified_params_dict['Domain'] = self.deployed.domain
        if previous_deployed.copyTagsToSnapshot != self.deployed.copyTagsToSnapshot:
            modified_params_dict['CopyTagsToSnapshot'] = self.deployed.copyTagsToSnapshot
        if previous_deployed.monitoringInterval != self.deployed.monitoringInterval:
            modified_params_dict['MonitoringInterval'] = self.deployed.monitoringInterval
        if previous_deployed.port != self.deployed.port:
            modified_params_dict['DBPortNumber'] = self.deployed.port
        if previous_deployed.publiclyAccessible != self.deployed.publiclyAccessible:
            modified_params_dict['PubliclyAccessible'] = self.deployed.publiclyAccessible
        if previous_deployed.monitoringRoleArn != self.deployed.monitoringRoleArn:
            modified_params_dict['MonitoringRoleArn'] = self.deployed.monitoringRoleArn
        if previous_deployed.domainIAMRoleName != self.deployed.domainIAMRoleName:
            modified_params_dict['DomainIAMRoleName'] = self.deployed.domainIAMRoleName
        if previous_deployed.promotionTier != self.deployed.promotionTier:
            modified_params_dict['PromotionTier'] = self.deployed.promotionTier
        if previous_deployed.enableIAMDatabaseAuthentication != self.deployed.enableIAMDatabaseAuthentication:
            modified_params_dict['EnableIAMDatabaseAuthentication'] = self.deployed.enableIAMDatabaseAuthentication

        not_null_params_dict = {k: v for k, v in modified_params_dict.iteritems() if v is not None}
        return not_null_params_dict

