from elb_helper import ElbHelper
from elb_utils import ElbUtils
from elb_validator import ElbValidator


class ElbService:
    def __init__(self, deployed, previous_deployed):
        self.deployed = deployed
        self.previous_deployed = previous_deployed
        self.elb_helper = ElbHelper(deployed)
        self.validator = ElbValidator(deployed=deployed, previous_deployed=previous_deployed)
        self.elb_utils = ElbUtils(deployed)

    def add_attributes(self):
        if self.validator.should_add_attributes():
            print "Configuring properties of ELB %s on %s" % (
                self.deployed.loadBalancerName, self.deployed.container.name)
            self.elb_helper.add_attributes(self.deployed)

    def modify_attributes(self):
        if self.validator.should_modify_attributes():
            print "Modifying properties of ELB %s on %s" % (
                self.deployed.loadBalancerName, self.deployed.container.name)
            self.elb_helper.add_attributes(self.deployed)

    def delete_stickiness_policies(self):
        policy_names = self.__get_all_stickiness_policy_names()
        if policy_names:
            print "Deleting policies %s from ELB %s on %s" % (
                policy_names, self.previous_deployed.loadBalancerName, self.previous_deployed.container.name)
        self.elb_helper.delete_policies(self.previous_deployed.loadBalancerName, policy_names)

    def create_app_cookie_stickiness(self):
        if self.validator.should_create_app_stickiness_policy(self.deployed.listeners):
            listeners_with_app_cookie_stickiness = self.__add_app_cookie_stickiness_policy_name_to_listeners(
                self.__get_listeners_with_app_cookie_stickiness_enabled(self.deployed.listeners)
            )
            listeners_pretty_printed = self.elb_utils.get_listeners_pretty_printed(listeners_with_app_cookie_stickiness)
            print "Creating app cookie stickiness for listeners %s" % listeners_pretty_printed

            for listener in listeners_with_app_cookie_stickiness:
                self.__apply_app_cookie_stickiness_policy(load_balancer_name=self.deployed.loadBalancerName,
                                                          listener=listener)

    def create_lb_cookie_stickiness(self):
        if self.validator.should_create_lb_stickiness_policy(self.deployed.listeners):
            listeners_with_lb_cookie_stickiness = self.__add_lb_cookie_stickiness_policy_name_to_listeners(
                self.__get_listeners_with_lb_cookie_stickiness_policy_enabled(self.deployed.listeners)
            )
            listeners_pretty_printed = self.elb_utils.get_listeners_pretty_printed(listeners_with_lb_cookie_stickiness)

            print "Creating load balancer cookie stickiness for listeners %s" % listeners_pretty_printed

            for listener in listeners_with_lb_cookie_stickiness:
                self.__apply_lb_cookie_stickiness_policy(load_balancer_name=self.deployed.loadBalancerName,
                                                         listener=listener)

    def attach_subnets(self):
        if self.validator.should_attach_subnets():
            subnets_to_be_attached = self.elb_utils.get_delta(set1=self.deployed.subnetIds,
                                                              set2=self.previous_deployed.subnetIds)
            subnets_pretty_printed = self.elb_utils.get_collection_pretty_printed(subnets_to_be_attached)
            print "Attaching subnets %s to ELB %s on %s" % (
                subnets_pretty_printed, self.deployed.loadBalancerName,
                self.deployed.container.name)
            self.elb_helper.attach_subnets(load_balancer_name=self.deployed.loadBalancerName,
                                           subnets=subnets_to_be_attached)

    def detach_subnets(self):
        if self.validator.should_detach_subnets():
            subnets_to_be_detached = self.elb_utils.get_delta(set1=self.previous_deployed.subnetIds,
                                                              set2=self.deployed.subnetIds)
            subnets_pretty_printed = self.elb_utils.get_collection_pretty_printed(subnets_to_be_detached)
            print "Detaching subnet(s) {} from ELB {} on {}".format(
                subnets_pretty_printed, self.deployed.loadBalancerName,
                self.deployed.container.name)
            self.elb_helper.detach_subnets(load_balancer_name=self.deployed.loadBalancerName,
                                           subnets=subnets_to_be_detached)
            self.__detach_associated_nis(subnets_to_be_detached)

    def attach_security_groups(self):
        if self.validator.should_attach_security_groups():
            security_groups_to_be_attached = self.deployed.securityGroups
            security_groups_pretty_printed = self.elb_utils.get_collection_pretty_printed(
                security_groups_to_be_attached)
            print "Attaching security groups {} to ELB {} on {}".format(
                security_groups_pretty_printed,
                self.deployed.loadBalancerName, self.deployed.container.name)
            self.elb_helper.modify_security_groups(load_balancer_name=self.deployed.loadBalancerName,
                                                   security_groups=security_groups_to_be_attached)

    def detach_security_groups(self):
        if self.validator.should_detach_security_groups():
            remaining_security_groups = self.elb_utils.get_common_elements(set1=self.deployed.securityGroups,
                                                                           set2=self.previous_deployed.securityGroups)
            security_groups_pretty_printed = self.elb_utils.get_collection_pretty_printed(remaining_security_groups)
            print "Modifying ELB %s to have security groups %s on %s" % (
                self.deployed.loadBalancerName, security_groups_pretty_printed,
                self.deployed.container.name)
            self.elb_helper.modify_security_groups(load_balancer_name=self.deployed.loadBalancerName,
                                                   security_groups=remaining_security_groups)

    def detach_availability_zones(self):
        if self.validator.should_detach_availability_zones():
            availability_zones_to_be_detached = self.elb_utils.get_delta(set1=self.previous_deployed.availabilityZones,
                                                                         set2=self.deployed.availabilityZones)
            availability_zones_pretty_printed = self.elb_utils.get_collection_pretty_printed(
                availability_zones_to_be_detached)
            print "Detaching availability zones %s from ELB %s on %s" % (
                availability_zones_pretty_printed,
                self.deployed.loadBalancerName, self.deployed.container.name)
            self.elb_helper.detach_availability_zones(load_balancer_name=self.deployed.loadBalancerName,
                                                      availability_zones=availability_zones_to_be_detached)

    def attach_availability_zones(self):
        if self.validator.should_attach_availability_zones():
            availability_zones_to_be_attached = self.elb_utils.get_delta(set1=self.deployed.availabilityZones,
                                                                         set2=self.previous_deployed.availabilityZones)
            availability_zones_pretty_printed = self.elb_utils.get_collection_pretty_printed(
                availability_zones_to_be_attached)
            print "Attaching availability zones %s from ELB %s on %s" % (
                availability_zones_pretty_printed,
                self.deployed.loadBalancerName, self.deployed.container.name)
            self.elb_helper.attach_availability_zones(load_balancer_name=self.deployed.loadBalancerName,
                                                      availability_zones=availability_zones_to_be_attached)

    def detach_ec2_instances(self):
        if self.validator.should_detach_ec2_instances():
            instances_to_be_detached = self.elb_utils.get_delta(set1=self.previous_deployed.ec2InstanceIds,
                                                                set2=self.deployed.ec2InstanceIds)
            instances_pretty_printed = self.elb_utils.get_collection_pretty_printed(instances_to_be_detached)
            print "Detaching instances %s from ELB %s on %s" % (
                instances_pretty_printed, self.deployed.loadBalancerName,
                self.deployed.container.name)
            self.elb_helper.deregister_ec2_instances(load_balancer_name=self.deployed.loadBalancerName,
                                                     instance_ids=instances_to_be_detached)

    def attach_ec2_instances(self):
        if self.validator.should_attach_ec2_instances():
            instances_to_be_attached = self.elb_utils.get_delta(set1=self.deployed.ec2InstanceIds,
                                                                set2=self.previous_deployed.ec2InstanceIds) if bool(
                self.previous_deployed) else self.deployed.ec2InstanceIds
            instances_pretty_printed = self.elb_utils.get_collection_pretty_printed(instances_to_be_attached)
            print "Attaching instances %s from ELB %s on %s" % (
                instances_pretty_printed, self.deployed.loadBalancerName,
                self.deployed.container.name)
            self.elb_helper.register_ec2_instances(load_balancer_name=self.deployed.loadBalancerName,
                                                   instance_ids=instances_to_be_attached)

    def delete_listeners(self):
        self.create_dummy_listener()
        listeners_pretty_printed = self.elb_utils.get_listeners_pretty_printed(self.previous_deployed.listeners)
        print "Deleting listeners %s from ELB %s on %s" % (
            listeners_pretty_printed,
            self.previous_deployed.loadBalancerName, self.previous_deployed.container.name)
        detached_load_balancer_ports = map(lambda listener: listener.loadBalancerPort, self.previous_deployed.listeners)
        self.elb_helper.detach_listeners(load_balancer_name=self.previous_deployed.loadBalancerName,
                                         load_balancer_ports=detached_load_balancer_ports)
        self.delete_stickiness_policies()

    def create_dummy_listener(self):
        print "Creating dummy listener %s" % self.elb_utils.get_dummy_listener_pretty_printed()
        self.elb_helper.attach_listeners(load_balancer_name=self.deployed.loadBalancerName,
                                         listener_dictionaries=[self.elb_utils.get_dummy_listener()])

    def delete_dummy_listener(self):
        print "Deleting dummy listener %s" % self.elb_utils.get_dummy_listener_pretty_printed()
        self.elb_helper.detach_listeners(load_balancer_name=self.deployed.loadBalancerName,
                                         load_balancer_ports=[self.elb_utils.get_dummy_listener()['LoadBalancerPort']])

    def create_listeners(self):
        self.delete_dummy_listener()
        listener_dictionary_to_be_attached = map(
            lambda listener: self.elb_utils.get_listener_as_dictionary(listener), self.deployed.listeners
        )
        listeners_pretty_printed = self.elb_utils.get_listeners_pretty_printed(self.deployed.listeners)
        print "Creating listeners %s for ELB %s on %s" % (
            listeners_pretty_printed, self.deployed.loadBalancerName,
            self.deployed.container.name)
        self.elb_helper.attach_listeners(load_balancer_name=self.deployed.loadBalancerName,
                                         listener_dictionaries=listener_dictionary_to_be_attached)
        self.create_app_cookie_stickiness()
        self.create_lb_cookie_stickiness()

    def create_elb(self):
        self.elb_helper.create(self.deployed)

    def delete_elb(self):
        self.elb_helper.delete(self.previous_deployed)

    def __detach_associated_nis(self, subnets_to_be_detached):
        associated_network_interfaces = self.elb_helper.fetch_network_interfaces_for_subnets(subnet_ids=subnets_to_be_detached,
                                                                                             load_balancer_name=self.deployed.loadBalancerName)
        if bool(associated_network_interfaces):
            self.elb_helper.remove_associated_network_interfaces(network_interface_ids=associated_network_interfaces,
                                                                 max_retries=self.deployed.maxRetriesForNiDetachment)

    def __apply_app_cookie_stickiness_policy(self, load_balancer_name, listener):
        self.elb_helper.add_app_cookie_policy_to_elb(listener=listener,
                                                     load_balancer_name=load_balancer_name)

        self.elb_helper.apply_policy_to_listener(load_balancer_port=listener.loadBalancerPort,
                                                 load_balancer_name=load_balancer_name,
                                                 policy_name=listener.appCookiePolicyName)

    def __apply_lb_cookie_stickiness_policy(self, load_balancer_name, listener):
        self.elb_helper.add_lb_cookie_policy_to_elb(listener=listener, load_balancer_name=load_balancer_name)

        self.elb_helper.apply_policy_to_listener(load_balancer_port=listener.loadBalancerPort,
                                                 load_balancer_name=load_balancer_name,
                                                 policy_name=listener.lbCookiePolicyName)

    def __get_app_cookie_stickiness_policy_names(self, listeners):
        return map(lambda listener: listener.appCookiePolicyName,
                   self.__get_listeners_with_app_cookie_stickiness_enabled(listeners))

    def __get_lb_cookie_stickiness_policy_names(self, listeners):
        return map(lambda listener: listener.lbCookiePolicyName,
                   self.__get_listeners_with_lb_cookie_stickiness_policy_enabled(listeners))

    def __add_app_cookie_stickiness_policy_name_to_listeners(self, listeners):
        for listener in listeners:
            listener.appCookiePolicyName = self.elb_utils.generate_policy_name()
        return listeners

    def __add_lb_cookie_stickiness_policy_name_to_listeners(self, listeners):
        for listener in listeners:
            listener.lbCookiePolicyName = self.elb_utils.generate_policy_name()
        return listeners

    def __get_all_stickiness_policy_names(self):
        return self.__get_app_cookie_stickiness_policy_names(self.previous_deployed.listeners) \
               + self.__get_lb_cookie_stickiness_policy_names(self.previous_deployed.listeners)

    def __get_listeners_with_app_cookie_stickiness_enabled(self, listeners):
        return filter(lambda listener: self.validator.is_valid_app_cookie_policy(listener), listeners)

    def __get_listeners_with_lb_cookie_stickiness_policy_enabled(self, listeners):
        return filter(lambda listener: bool(listener.enableLbCookieStickiness), listeners)
