from commons.aws_helper import AWSHelper
from ec2.ec2_helper import EC2Helper
from ec2.ni.ni_helper import NIHelper
from elb_utils import ElbUtils


class ElbHelper(AWSHelper):
    def __init__(self, deployed):
        super(ElbHelper, self).__init__(deployed)
        self.elb_client = self.get_aws_client(region=deployed.region, resource_name='elb')
        self.elb_utils = ElbUtils(deployed)
        self.ec2_helper = EC2Helper(deployed)
        self.ni_helper = NIHelper(deployed)

    def create(self, deployed):
        return self.elb_client.create_load_balancer(
            Listeners=map(lambda ls: self.elb_utils.get_listener_as_dictionary(ls), deployed.listeners),
            LoadBalancerName=deployed.loadBalancerName,
            AvailabilityZones=map(lambda availability_zone: availability_zone, deployed.availabilityZones),
            SecurityGroups=self.ec2_helper.get_security_group_id_list(deployed.securityGroups),
            Subnets=self.get_subnet_ids(deployed.subnetIds),
            Scheme=deployed.scheme
        )

    def add_attributes(self, deployed):
        attributes = self.__prepare_load_balancer_attributes(deployed)
        return self.elb_client.modify_load_balancer_attributes(
            LoadBalancerName=deployed.loadBalancerName,
            LoadBalancerAttributes=attributes
        )

    def register_ec2_instances(self, load_balancer_name, instance_ids):
        return self.elb_client.register_instances_with_load_balancer(
            Instances=self.__get_ec2_instances_dictionary(instance_ids),
            LoadBalancerName=load_balancer_name,
        )

    def add_app_cookie_policy_to_elb(self, listener, load_balancer_name):
        return self.elb_client.create_app_cookie_stickiness_policy(
            CookieName=listener.appCookieName,
            LoadBalancerName=load_balancer_name,
            PolicyName=listener.appCookiePolicyName,
        )

    def add_lb_cookie_policy_to_elb(self, listener, load_balancer_name):
        if listener.lbCookieExpirationPeriod > 0:
            response = self.elb_client.create_lb_cookie_stickiness_policy(
                LoadBalancerName=load_balancer_name,
                PolicyName=listener.lbCookiePolicyName,
                CookieExpirationPeriod=listener.lbCookieExpirationPeriod
            )
        else:
            response = self.elb_client.create_lb_cookie_stickiness_policy(
                LoadBalancerName=load_balancer_name,
                PolicyName=listener.lbCookiePolicyName
            )
        return response

    def apply_policy_to_listener(self, load_balancer_port, load_balancer_name, policy_name):
        return self.elb_client.set_load_balancer_policies_of_listener(
            LoadBalancerName=load_balancer_name,
            LoadBalancerPort=load_balancer_port,
            PolicyNames=[policy_name]
        )

    def delete_policies(self, load_balancer_name, policy_names):
        return [self.delete_policy(load_balancer_name=load_balancer_name, policy_name=policy_name) for policy_name in
                policy_names]

    def delete_policy(self, load_balancer_name, policy_name):
        return self.elb_client.delete_load_balancer_policy(
            LoadBalancerName=load_balancer_name,
            PolicyName=policy_name,
        )

    def delete_app_cookie_policy(self, deployed):
        listeners_with_app_cookie_policy = filter(lambda listener: bool(listener.appCookiePolicyName.strip()),
                                                  deployed.listeners)
        for listener in listeners_with_app_cookie_policy:
            self.delete_policy(load_balancer_name=deployed.loadBalancerName, policy_name=listener.appCookiePolicyName)

    def delete_lb_cookie_policy(self, deployed):
        listeners_with_lb_cookie_policy = filter(lambda listener: bool(listener.lbCookiePolicyName.strip()),
                                                 deployed.listeners)
        for listener in listeners_with_lb_cookie_policy:
            self.delete_policy(load_balancer_name=deployed.loadBalancerName, policy_name=listener.lbCookiePolicyName)

    def delete(self, deployed):
        return self.elb_client.delete_load_balancer(
            LoadBalancerName=deployed.loadBalancerName
        )

    def modify_security_groups(self, load_balancer_name, security_groups):
        return self.elb_client.apply_security_groups_to_load_balancer(
            LoadBalancerName=load_balancer_name,
            SecurityGroups=self.ec2_helper.get_security_group_id_list(security_groups)
        )

    def detach_listeners(self, load_balancer_name, load_balancer_ports):
        return self.elb_client.delete_load_balancer_listeners(
            LoadBalancerName=load_balancer_name,
            LoadBalancerPorts=load_balancer_ports
        )

    def attach_listeners(self, load_balancer_name, listener_dictionaries):
        return self.elb_client.create_load_balancer_listeners(
            LoadBalancerName=load_balancer_name,
            Listeners=listener_dictionaries
        )

    def attach_subnets(self, load_balancer_name, subnets):
        return self.elb_client.attach_load_balancer_to_subnets(
            LoadBalancerName=load_balancer_name,
            Subnets=self.get_subnet_ids(subnets)
        )

    def detach_subnets(self, load_balancer_name, subnets):
        return self.elb_client.detach_load_balancer_from_subnets(
            LoadBalancerName=load_balancer_name,
            Subnets=self.get_subnet_ids(subnets)
        )

    def attach_availability_zones(self, load_balancer_name, availability_zones):
        return self.elb_client.enable_availability_zones_for_load_balancer(
            LoadBalancerName=load_balancer_name,
            AvailabilityZones=map(lambda zone: zone, availability_zones)
        )

    def detach_availability_zones(self, load_balancer_name, availability_zones):
        return self.elb_client.disable_availability_zones_for_load_balancer(
            LoadBalancerName=load_balancer_name,
            AvailabilityZones=map(lambda zone: zone, availability_zones)
        )

    def deregister_ec2_instances(self, load_balancer_name, instance_ids):
        return self.elb_client.deregister_instances_from_load_balancer(
            Instances=self.__get_ec2_instances_dictionary(instance_ids),
            LoadBalancerName=load_balancer_name,
        )

    def fetch_network_interfaces_for_subnets(self, subnet_ids, load_balancer_name):
        return filter(lambda ni: bool(ni),
                      reduce(lambda first_ni, next_ni: first_ni.extend(next_ni),
                             map(lambda subnet_id: self.__get_network_interface_ids_for_subnet(subnet_id=subnet_id,
                                                                                               load_balancer_name=load_balancer_name),
                                 subnet_ids)))

    def __get_network_interface_ids_for_subnet(self, subnet_id, load_balancer_name):
        return self.elb_utils.get_values_from_dictionaries(
            dict=self.__fetch_network_interfaces_for_subnet(subnet_id=subnet_id, load_balancer_name=load_balancer_name),
            property_name='NetworkInterfaceId')

    def __fetch_network_interfaces_for_subnet(self, subnet_id, load_balancer_name):
        criteria = [{"Name": "subnet-id", "Values": [self.get_subnet_id(subnet_id)]},
                    {"Name": "description", "Values": ["ELB {}".format(load_balancer_name)]}]
        associated_network_interfaces = self.ni_helper.fetch_ni_by_criteria(criteria)
        return associated_network_interfaces if bool(
            associated_network_interfaces and associated_network_interfaces[0]) else []

    def remove_associated_network_interfaces(self, network_interface_ids, max_retries):
        for network_interface_id in network_interface_ids:
            self.__remove_network_interface(network_interface_id=network_interface_id, max_retries=max_retries)

    def get_subnet_ids(self, subnets):
        return map(lambda subnet: self.get_subnet_id(subnet) if self.is_starts_with_name(
            property_value=subnet) else subnet, subnets)

    def get_subnet_id(self, subnet):
        return (subnet, self.ec2_helper.get_subnet_id_by_name(self.get_property_name(subnet)))[
            self.is_starts_with_name(subnet)]

    def __remove_network_interface(self, network_interface_id, max_retries):
        print "Detaching ni {}".format(network_interface_id)
        if self.ni_helper.detach_ni(network_interface_id):
            self.ni_helper.wait_for_ni_to_be_detached(ni_id=network_interface_id, max_retries=max_retries)
        self.ni_helper.delete_ni_by_id(network_interface_id)

    def __get_ec2_instance_ids(self, instance_ids):
        return map(lambda instance: self.ec2_helper.get_instance_id_by_id_or_name(instance), instance_ids)

    def __get_ec2_instances_dictionary(self, instance_ids):
        return map(lambda instance_id: {'InstanceId': instance_id},
                   self.__get_ec2_instance_ids(instance_ids))

    def __prepare_load_balancer_attributes(self, deployed):
        params = {}
        self.__add_connection_settings(params=params, deployed=deployed)
        self.__add_access_log_params(params=params, deployed=deployed)
        self.__add_cross_zone_load_balancing(params=params, deployed=deployed)
        self.__add_connection_draining(params=params, deployed=deployed)
        return params

    def __add_cross_zone_load_balancing(self, params, deployed):
        params['CrossZoneLoadBalancing'] = {
            'Enabled': deployed.enableCrossZoneLoadBalancing
        }

    def __add_access_log_params(self, params, deployed):
        if deployed.enableAccessLog:
            params['AccessLog'] = self.elb_utils.filter_empty_params({
                'Enabled': deployed.enableAccessLog,
                'S3BucketName': deployed.accessLogS3BucketName,
                'EmitInterval': deployed.accessLogEmitInterval,
                'S3BucketPrefix': deployed.accessLogS3BucketPrefix
            })
        else:
            params['AccessLog'] = {'Enabled': deployed.enableAccessLog}

    def __add_connection_draining(self, params, deployed):
        if deployed.enableConnectionDraining:
            params['ConnectionDraining'] = self.elb_utils.filter_empty_params({
                'Enabled': deployed.enableConnectionDraining,
                'Timeout': deployed.connectionDrainingTimeout
            })
        else:
            params['ConnectionDraining'] = {'Enabled': deployed.enableConnectionDraining}

    def __add_connection_settings(self, params, deployed):
        if deployed.connectionSettingIdleTimeout:
            params['ConnectionSettings'] = {
                'IdleTimeout': deployed.connectionSettingIdleTimeout
            }
