# -*- coding: utf-8 -*-
import hashlib

from commons.aws_helper import AWSHelper
import mimetypes
import os
from datetime import datetime
import dateutil.tz
from six import print_


class S3Helper(AWSHelper):
    def __init__(self, deployed):
        super(S3Helper, self).__init__(deployed.container)
        self.s3_client = self.get_aws_client(resource_name='s3', region=deployed.container.region)
        self.deployed = deployed


    def upload_file_on_bucket(self, key, acl, file):
        mime_type = mimetypes.guess_type(file.name)[0]
        return self.upload_file(acl, file, key, mime_type)

    def upload_file(self, acl, file, key, mime_type=None):
        upload_params = {'ACL': acl,
                         'Bucket': self.deployed.container.bucketName,
                         'Key': key,
                         'Body': file}
        if mime_type:
            upload_params.update({'ContentType': mime_type})
        return self.s3_client.put_object(**upload_params)

    def delete_file_from_bucket(self, key):
        hosting_response = self.s3_client.delete_object(
            Bucket=self.deployed.container.bucketName, Key=key)
        return hosting_response

    def sync_folder(self):

        print("Syncing S3 folder with deployable folder")
        s3_bucket = self.deployed.container.bucketName
        local_folder = self.deployed.file.path

        s3_path = ""
        if self.deployed.targetPath:
            s3_path = self.deployed.targetPath
            if self.deployed.targetPath.endswith("/"):
                s3_path = self.deployed.targetPath[0: -1]
            if self.deployed.targetPath.startswith("/"):
                s3_path = self.deployed.targetPath[1:]

        s3_files = {}
        paginator = self.s3_client.get_paginator('list_objects_v2')
        for page in paginator.paginate(Bucket=s3_bucket, Prefix=s3_path):
            for file in page.get('Contents', []):
                s3_files[os.path.relpath(file['Key'], s3_path)] = file

        print("Remote files information fetched from S3")

        new_files = []
        local_files = []
        removed_files = []
        modified_files = []
        local_file_count = 0

        # Traverse local folder and compare against retrieved s3 files
        for path, subdirectories, filenames in os.walk(local_folder):
            for name in filenames:
                local_file_count = local_file_count + 1
                abs_path = os.path.join(path, name)
                key = os.path.relpath(abs_path.__str__(), local_folder)
                local_files.append(key)

                if key in s3_files:
                    # Files exists in local and remote

                    #Checking for hash
                    s3_file = s3_files[key]
                    if s3_file['Size'] != os.path.getsize(abs_path):
                        # Local file size has changed
                        if not abs_path in modified_files:
                            modified_files.append(abs_path)
                    else:
                        # Check for checksum
                        s3_file_checksum = s3_file['ETag']
                        if s3_file_checksum.startswith('"'):
                            s3_file_checksum = s3_file_checksum[1:]
                        if s3_file_checksum.endswith('"'):
                            s3_file_checksum = s3_file_checksum[:-1]
                        local_file_checksum = self.compute_md5_checksum(abs_path)

                        if s3_file_checksum != local_file_checksum:
                            modified_files.append(abs_path)

                else:
                    # New file in local
                    new_files.append(abs_path)

        for key in s3_files:
            if key not in local_files:
                # File has been removed from local
                removed_files.append(os.path.join(local_folder, key.__str__()))

        print("Total number of deployable files: ", local_file_count)

        index = 1
        for abs_path in modified_files:
            key = os.path.relpath(abs_path.__str__(), local_folder)
            if s3_path:
                key = s3_path + '/' + key
            print("Uploading modified file: ", index, " of ", len(modified_files), " : ",  key)
            self.s3_client.upload_file(abs_path, s3_bucket, key)
            index = index + 1

        index = 1
        for abs_path in new_files:
            key = os.path.relpath(abs_path.__str__(), local_folder)
            if s3_path:
                key = s3_path + '/' + key
            print("Uploading new file: ", index, " of ", len(new_files), " : ", key)
            self.s3_client.upload_file(abs_path, s3_bucket, key)
            index = index + 1

        if self.deployed.removeDeletedFiles:
            index = 1
            for abs_path in removed_files:
                key = os.path.relpath(abs_path.__str__(), local_folder)
                if s3_path:
                    key = s3_path + '/' + key
                print("Deleting removed file: ", index, " of ", len(removed_files), " : ", key)
                self.s3_client.delete_object(Bucket=s3_bucket, Key= key)
                index = index + 1

    def compute_md5_checksum(self, file_path):
        with open(file_path, 'rb') as file:
            file_content = file.read()
            checksum = hashlib.md5(file_content).hexdigest()
            return checksum
