from aws_lambda.lambda_helper import LambdaHelper
from commons.aws_helper import AWSHelper


class APIHelper(AWSHelper):
    def __init__(self, deployed):
        super(APIHelper, self).__init__(deployed)
        self.client = self.get_aws_client(region=deployed.region, resource_name='apigateway')

    def str_to_bool(self, v):
        return v.lower() in ("yes", "true", "t", "1")

    def create_api(self):
        params = {'name': APIHelper.get_api_name(self.deployed),
                  'version': self.deployed.version,
                  'description': self.deployed.description}
        params = self.remove_none_keys(params)
        print "Creating REST API {0} on {1}".format(
            APIHelper.get_api_name(self.deployed), self.deployed.container.name)
        rest_api = self.client.create_rest_api(**params)
        self.deployed.apiId = rest_api['id']
        print "Created REST API {0} with id {1} on {2}".format(
            APIHelper.get_api_name(self.deployed),
            self.deployed.apiId, self.deployed.container.name)
        all_resources = self.client.get_resources(restApiId=self.deployed.apiId)
        self.deployed.rootResourceId = all_resources['items'][0]['id']
        print "Root Resource implicitly created with id {0}".format(self.deployed.rootResourceId)

    def create_resource(self, parent_resource_id, resource):
        print "Creating resource {0} as child of resource {1}".format(resource.name, parent_resource_id)
        response = self.client.create_resource(
            restApiId=self.deployed.apiId,
            parentId=parent_resource_id,
            pathPart=resource.path
        )
        if self.is_success(response):
            resource.resourceId = response['id']
            resource.fullPath = response['path']
            print "Created resource {0} as child of resource {1}".format(resource.name, parent_resource_id)
        else:
            raise Exception(
                "Could not create resource {0} as child of resource {1}".format(resource.name, parent_resource_id))

    def create_method(self, resource, method):
        APIHelper.validate_method(method)
        request_params = {'method.request.%s' % k: self.str_to_bool(v) for k, v in
                          method.requestParameters.iteritems()} if method.requestParameters else None
        params = {'restApiId': self.deployed.apiId, 'resourceId': resource.resourceId, 'httpMethod': method.httpMethod,
                  'authorizationType': method.authorizationType, 'apiKeyRequired': method.apiKeyRequired,
                  'operationName': method.operationName, 'requestParameters': request_params,
                  'requestModels': method.requestModels}
        params = self.remove_none_keys(params)
        print "Creating method {0} on resource {1}.".format(method.httpMethod, resource.name)
        method_response = self.client.put_method(**params)
        if self.is_success(method_response):
            print "Created method {0} on resource {1}.".format(method.httpMethod, resource.name)
        else:
            raise Exception("Could not create method {0} on resource {1}".format(method.httpMethod, resource.name))

    def save_url(self, resource, method):
        method.methodURL = "https://{0}.execute-api.{1}.amazonaws.com/{2}{3}".format(self.deployed.apiId,
                                                                                     self.deployed.region,
                                                                                     self.deployed.stage,
                                                                                     resource.fullPath)

    @staticmethod
    def validate_method(method):
        if method.authorizationType == 'CUSTOM' and not method.authorizerId:
            raise Exception("Authorizer ID is mandatory for Authorization Type 'CUSTOM'")

    def create_integration(self, resource, method):
        params = {'restApiId': self.deployed.apiId, 'resourceId': resource.resourceId, 'type': method.integrationType,
                  'integrationHttpMethod': method.integrationHttpMethod, 'uri': self.__get_uri(method),
                  'credentials': method.credentials, 'requestParameters': method.integrationRequestParameters,
                  'httpMethod': method.httpMethod, 'requestTemplates': method.requestTemplates,
                  'contentHandling': method.contentHandling
                  }
        params = self.remove_none_keys(params)
        print "Creating integration {0} on resource {1}.".format(method.httpMethod, resource.name)
        integration_response = self.client.put_integration(**params)
        if self.is_success(integration_response):
            print "Created integration {0} on resource {1}.".format(method.httpMethod, resource.name)
        else:
            raise Exception("Could not create integration {0} on resource {1}".format(method.httpMethod, resource.name))

    def create_method_response(self, resource, method, response_mapping):
        method_response_params = {'restApiId': self.deployed.apiId, 'resourceId': resource.resourceId,
                                  'httpMethod': method.httpMethod, 'statusCode': response_mapping.statusCode,
                                  'responseParameters': self.__transform_response_parameters(
                                      response_mapping.responseParameters),
                                  'responseModels': response_mapping.responseModels
                                  }
        method_response_params = self.remove_none_keys(method_response_params)
        print "Creating method response for status code {0} for method {1} on resource {2}." \
            .format(response_mapping.statusCode, method.httpMethod, resource.name)
        method_response_response = self.client.put_method_response(**method_response_params)
        if self.is_success(method_response_response):
            print "Created method response for status code {0} for method {1} on resource {2}." \
                .format(response_mapping.statusCode, method.httpMethod, resource.name)
        else:
            raise Exception("Could not create integration {0} on resource {1}".format(method.httpMethod, resource.name))

    def __transform_response_parameters(self, dict):
        params = {}
        for key in dict.keys():
            params[key] = self.str_to_bool(dict[key])
        return params

    def create_response_mapping(self, resource, method, response_mapping):
        self.create_method_response(resource, method, response_mapping)
        self.create_integration_response(resource, method, response_mapping)

    def create_integration_response(self, resource, method, response_mapping):
        integration_response_params = {'restApiId': self.deployed.apiId, 'resourceId': resource.resourceId,
                                       'httpMethod': method.httpMethod, 'statusCode': response_mapping.statusCode,
                                       'selectionPattern': response_mapping.selectionPattern,
                                       'responseParameters': response_mapping.integrationResponseParameters,
                                       'responseTemplates': response_mapping.responseTemplates
                                       }
        integration_response_params = self.remove_none_keys(integration_response_params)
        print "Creating integration response for status code {0} for method {1} on resource {2}." \
            .format(response_mapping.statusCode, method.httpMethod, resource.name)
        method_response_response = self.client.put_integration_response(**integration_response_params)
        if self.is_success(method_response_response):
            print "Created integration response for status code {0} for method {1} on resource {2}." \
                .format(response_mapping.statusCode, method.httpMethod, resource.name)
        else:
            raise Exception("Could not create integration {0} on resource {1}".format(method.httpMethod, resource.name))

    @staticmethod
    def remove_none_keys(dictionary):
        return {k: v for k, v in dictionary.iteritems() if v is not None}

    @staticmethod
    def get_ordered_list(resources):
        all_resources = set(resources)
        ordered_resources = []
        while all_resources:
            resources_parent_included = [resource for resource in all_resources if
                                         APIHelper.parent_included(ordered_resources, resource)]
            if not resources_parent_included:
                raise Exception("Parent of certain resources not found")
            ordered_resources.extend(resources_parent_included)
            all_resources = all_resources.difference(resources_parent_included)
        return ordered_resources

    @staticmethod
    def parent_included(resources, resource):
        return APIHelper.is_first_level_resource(resource) or not APIHelper.is_starts_with_name(
            resource.parent) or resource.parent[5:] in [resource.name for resource in resources]

    @staticmethod
    def is_first_level_resource(resource):
        return not bool(resource.parent)

    @staticmethod
    def get_parent_resource_id(deployed, resource, all_resources):
        parent_resource_id = deployed.rootResourceId if APIHelper.is_first_level_resource(resource) \
            else APIHelper.get_parent_resource_id_by_name(resource, all_resources)
        # parent property represents the actual parent resource id
        parent_resource_id = resource.parent if not parent_resource_id else parent_resource_id
        return parent_resource_id

    @staticmethod
    def get_parent_resource_id_by_name(resource, all_resources):
        parent = resource.parent
        if parent and APIHelper.is_starts_with_name(parent):
            return next((resource.resourceId for resource in all_resources if resource.name == parent[5:]), None)
        else:
            return None

    def destroy_rest_api(self, previous_deployed):
        print "Destroying REST API {0} from {1}.".format(
            APIHelper.get_api_name(previous_deployed),
            previous_deployed.container.name)
        delete_response = self.client.delete_rest_api(
            restApiId=previous_deployed.apiId
        )
        if self.is_success(delete_response):
            print "Destroying REST API {0} from {1}.".format(
                APIHelper.get_api_name(previous_deployed),
                previous_deployed.container.name)
        else:
            raise Exception("Could not destroy REST API {0} from {1}.".format(
                APIHelper.get_api_name(self.deployed), self.deployed.container.name))

    def deploy_api(self):
        print "Deploying REST API {0} on {1}.".format(
            APIHelper.get_api_name(self.deployed),
            self.deployed.container.name)
        deployment_response = self.client.create_deployment(
            restApiId=self.deployed.apiId,
            stageName=self.deployed.stage
        )
        if self.is_success(deployment_response):
            self.deployed.deploymentId = deployment_response['id']
            print "Deployed REST API {0} on {1}.".format(
                APIHelper.get_api_name(self.deployed),
                self.deployed.container.name)
        else:
            raise Exception(
                "Could not deploy REST API {0} on {1}.".format(
                    APIHelper.get_api_name(self.deployed),
                    self.deployed.container.name))

    def modify_api(self, previous_deployed, patches):
        print "Modifying REST API {0} on {1}.".format(
            APIHelper.get_api_name(previous_deployed),
            previous_deployed.container.name)
        deployment_response = self.client.update_rest_api(
            restApiId=previous_deployed.apiId,
            patchOperations=patches
        )
        if self.is_success(deployment_response):
            print "Modified REST API {0} on {1}.".format(
                APIHelper.get_api_name(previous_deployed),
                previous_deployed.container.name)
        else:
            raise Exception(
                "Could not modify REST API {0} on {1}.".format(
                    APIHelper.get_api_name(previous_deployed),
                    previous_deployed.container.name))

    def destroy_resource(self, previous_deployed, resource):
        print "Destroying resource {0}({1}) from REST API {2}".format(resource.name, resource.resourceId,
                                                                      APIHelper.get_api_name(previous_deployed))
        response = self.client.delete_resource(
            restApiId=previous_deployed.apiId,
            resourceId=resource.resourceId
        )
        if self.is_success(response):
            resource.resourceId = None
            resource.fullPath = None
            print "Destroyed resource {0} from REST API {1}".format(resource.name,
                                                                    APIHelper.get_api_name(previous_deployed))
        else:
            raise Exception("Could not destroy resource {0}".format(resource.name))

    @staticmethod
    def find_diff(previous_deployed, deployed):
        patches = []
        if APIHelper.get_api_name(previous_deployed) != APIHelper.get_api_name(deployed):
            patches.append({'op': 'replace', 'path': '/name', 'value': APIHelper.get_api_name(deployed)})
        if previous_deployed.version != deployed.version:
            patches.append({'op': 'replace', 'path': '/version', 'value': deployed.version})
        if previous_deployed.description != deployed.description:
            raise RuntimeError("Description of a rest api cannot be modified.")
        if previous_deployed.region != deployed.region:
            raise RuntimeError("Region of a rest api cannot be modified.")
        return patches

    @staticmethod
    def get_api_name(deployed):
        return deployed.apiName if bool(deployed.apiName) else deployed.name

    def undeploy_api(self, previous_deployed):
        print "Undeploying REST API {0} from {1}.".format(
            APIHelper.get_api_name(previous_deployed),
            previous_deployed.container.name)
        delete_deployment_response = self.client.delete_deployment(
            restApiId=previous_deployed.apiId,
            deploymentId=previous_deployed.deploymentId
        )
        if self.is_success(delete_deployment_response):
            print "Undeployed REST API {0} from {1}.".format(
                APIHelper.get_api_name(self.deployed),
                self.deployed.container.name)
        else:
            raise Exception("Could not deploy REST API {0} from {1}.".format(
                APIHelper.get_api_name(previous_deployed), self.deployed.container.name))

    def delete_stage(self, previous_deployed):
        print "Deleting stage {0} from REST API {1} on {2}.".format(previous_deployed.stage,
                                                                    APIHelper.get_api_name(previous_deployed),
                                                                    previous_deployed.container.name)
        delete_deployment_response = self.client.delete_stage(
            restApiId=previous_deployed.apiId,
            stageName=previous_deployed.stage
        )
        if self.is_success(delete_deployment_response):
            print "Deleted stage {0} from REST API {1} on {2}.".format(previous_deployed.stage,
                                                                       APIHelper.get_api_name(previous_deployed),
                                                                       previous_deployed.container.name)
        else:
            raise Exception("Could not delete stage {0} from REST API {1} on {2}.".format(previous_deployed.stage,
                                                                                          APIHelper.get_api_name(
                                                                                              previous_deployed),
                                                                                          previous_deployed.container.name))

    def __get_uri(self, method):
        uri = method.uri
        if self.is_starts_with_name(uri):
            lambda_name = self.get_property_name(method.uri)
            lambda_arn = LambdaHelper(self.deployed).get_lambda_function(lambda_name)['FunctionArn']
            uri = 'arn:aws:apigateway:{}:lambda:path/2015-03-31/functions/{}/invocations'.format(self.deployed.region,
                                                                                                 lambda_arn)

        return uri
