from commons.aws_helper import AWSHelper
from cloudformation.utils.cf_client import CFClient
from botocore.exceptions import ClientError

import re

class SCClient(object):
    def __init__(self, account, region):
        self.session = AWSHelper.create_session(account)
        AWSHelper.set_ca_bundle_path()
        self.sc_client = self.session.client('servicecatalog', region_name=region)
        self.cf_client = self.session.client('cloudformation', region_name=region)

    def list_provisioning_artifacts(self, deployed):
        data = self.sc_client.list_provisioning_artifacts(
            ProductId=deployed.productId if deployed.productId is not None else self.search_product_id(deployed)
        )
        return data

    def get_provisioning_artifact_id(self, deployed):
        data = self.list_provisioning_artifacts(deployed)
        for ppa in data['ProvisioningArtifactDetails']:
            if ppa['Name'] == deployed.productVersion:
                return ppa['Id']
        return None

    def describe_provisioning_product(self, provisioning_product_data):
        data = self.sc_client.describe_provisioned_product(
            Id=provisioning_product_data['RecordDetail']['ProvisionedProductId']
        )
        return data

    def search_product_id(self, deployed):
        data = self.sc_client.search_products(Filters={"FullTextSearch":["{}".format(deployed.productName)]})
        if len(data['ProductViewSummaries']) == 1:
            productId = data['ProductViewSummaries'][0]['ProductId']
            deployed.productId = productId
            return deployed.productId
        elif len(data['ProductViewSummaries']) > 1:
            product_ids = []
            for i in range(len(data['ProductViewSummaries'])):
                product_ids.append(data['ProductViewSummaries'][i]['ProductId'])
            raise Exception('More than one ProductId were found with productName {}: {}. You may explicitly specify productId property.'.format(deployed.productName, ", ".join(product_ids)))
        else:
            raise Exception('No Products were found with productName {}'.format(deployed.productName))

    def get_launch_path_id(self, deployed):
        data = self.sc_client.list_launch_paths(
            ProductId=deployed.productId if deployed.productId is not None else self.search_product_id(deployed)
        )
        if len(data['LaunchPathSummaries']) == 1:
            if data['LaunchPathSummaries'][0]['Name'] == deployed.portfolioName or deployed.portfolioName is None:
                pathId = data['LaunchPathSummaries'][0]['Id']
                deployed.pathId = pathId
                return deployed.pathId
            else:
                raise Exception('Specified Launch Path(Portfolio Name) {} is not associated with productName {}.'.format(deployed.portfolioName, deployed.productName))
        elif len(data['LaunchPathSummaries']) > 1:
            lp_ids = []
            for i in range(len(data['LaunchPathSummaries'])):
                lp_ids.append(data['LaunchPathSummaries'][i]['Name'])
            if deployed.portfolioName is None:
                raise Exception('More than one Launch Paths(Portfolios) were found with productName {}: {}. You may explicitly specify portfolioName property.'.format(deployed.productName, ", ".join(lp_ids)))
            elif deployed.portfolioName is not None and deployed.portfolioName not in lp_ids:
                raise Exception('Provided Launch Path(Portfolio) {} is not associated with productName {}. Available Launch Paths: {}. Please specify correct portfolioName property.'.format(deployed.portfolioName, deployed.productName, ", ".join(lp_ids)))
            else:
                pathId = data['LaunchPathSummaries'][lp_ids.index(deployed.portfolioName)]['Id']
                deployed.pathId = pathId
                return deployed.pathId
        else:
            raise Exception('No Launch Paths were found with productName {}.'.format(deployed.productName))

    def search_provisioned_products(self, provisionedProductId):
        return self.sc_client.search_provisioned_products(Filters={"SearchQuery":["id:{}".format(provisionedProductId)]})['ProvisionedProducts']

    def describe_record(self,provisioning_product_data):
        data = self.sc_client.describe_record(
            Id=provisioning_product_data['RecordDetail']['RecordId']
        )
        return data

    def list_resources(self, deployed, provisioning_product_data):
        stackname = self._sanatize_name(self.get_stack_name(provisioning_product_data))
        stackid = self.get_stackId(provisioning_product_data)

        if self._stack_exists(stackid):
            return self.cf_client.list_stack_resources(StackName=stackid)['StackResourceSummaries']
        return None

    def wait_for_provisioning_product(self, provisioning_product_data,deployed):
        ## maybe use waiter
        try:
            if self.get_stackId(provisioning_product_data):
                print "Events from Cloudformation Stack [%s]:" % self.get_stack_name(provisioning_product_data)
            while self.describe_provisioning_product(provisioning_product_data)['ProvisionedProductDetail']['Status'] != "AVAILABLE":
                provisionedProductDetail = self.describe_provisioning_product(provisioning_product_data)['ProvisionedProductDetail']
                if provisionedProductDetail['Status'] == "TAINTED":
                    print provisionedProductDetail['StatusMessage']
                    raise Exception("Failed to provision product [%s]" % (provisionedProductDetail['Name']))
                if  provisionedProductDetail['Status'] == "ERROR":
                    print provisionedProductDetail['StatusMessage']
                    self.delete_provisioned_product(deployed)
                    raise Exception("")
                self.print_events(deployed,provisioning_product_data)
                pass
        except Exception as e:
            print "{0}".format(str(e))
            raise

    def wait_for_updating_provisioning_product(self, provisioning_product_data,deployed,previousDeployed):
        ## maybe use waiter

        try:
            if self.get_stackId(provisioning_product_data):
                print "Events from Cloudformation Stack [%s]:" % self.get_stack_name(provisioning_product_data)
            while self.describe_provisioning_product(provisioning_product_data)['ProvisionedProductDetail']['Status'] != "AVAILABLE":
                provisionedProductDetail = self.describe_provisioning_product(provisioning_product_data)['ProvisionedProductDetail']
                if provisionedProductDetail['Status'] == "TAINTED":
                    print provisionedProductDetail['StatusMessage']
                    print "Reverting back to version %s" % previousDeployed.productVersion
                    self.update_provisioned_product(deployed,previousDeployed,True)
                    raise Exception("Failed to provision product [%s]" % (provisionedProductDetail['Name']))
                if  provisionedProductDetail['Status'] == "ERROR":
                    print provisionedProductDetail['StatusMessage']
                    raise Exception("")
                self.print_events(deployed,provisioning_product_data)
                pass
        except Exception as e:
            print "{0}".format(str(e))
            raise

    def wait_for_terminate_provisioned_product(self, deployed, data):

        if self.get_stackId(data) is None:

            raise Exception("No stack created because the product was deprovisioned due to failure")
        else:
            print "Events from Cloudformation Stack [%s]:" % self.get_stack_name(data)
        while len(self.search_provisioned_products(deployed.provisionedProductId)) > 0:
            self.print_events(deployed,data)
            pass


    def provision_product(self, deployed):
        try:
            data = self.sc_client.provision_product(
                ProductId=deployed.productId if deployed.productId is not None else self.search_product_id(deployed),
                PathId=self.get_launch_path_id(deployed),
                ProvisioningArtifactId=self.get_provisioning_artifact_id(deployed),
                ProvisionedProductName=deployed.provisionedProductName,
                ProvisioningParameters=self.prepare_provisioning_parameters(deployed) if len(deployed.provisioningParameters) > 0 else []
            )
            deployed.provisionedProductId = data['RecordDetail']['ProvisionedProductId']
        except Exception as error:
            raise Exception("There was an error during provisioning {} with version {}: {}".format(deployed.provisionedProductName, deployed.productVersion, error))
        else:
            print "Product %s is being launched with version %s" % (deployed.provisionedProductName, deployed.productVersion)
            self.wait_for_provisioning_product(data,deployed)
            self.print_resources(data, deployed)
            deployed.stackId = self.get_stackId(data)

    def update_provisioned_product(self, deployed, previousDeployed, rollback_update=False):
        try:
            if rollback_update:
                deployed = previousDeployed
            data = self.sc_client.update_provisioned_product(
                ProvisionedProductId=deployed.provisionedProductId,
                PathId=self.get_launch_path_id(deployed),
                ProductId=deployed.productId,
                ProvisioningArtifactId=self.get_provisioning_artifact_id(deployed),
                ProvisioningParameters=self.prepare_provisioning_parameters(deployed) if len(deployed.provisioningParameters) > 0 else []
            )
            if not rollback_update:
                print "Provisioned product %s is being updated to version %s" % (deployed.provisionedProductName, deployed.productVersion)
        except Exception as error:
            raise Exception("There was an error during updating {} to version {}: {}".format(deployed.provisionedProductName, deployed.productVersion, error))
        else:
            self.wait_for_updating_provisioning_product(data,deployed,previousDeployed)
            self.print_resources(data, deployed)
            deployed.stackId = self.get_stackId(data)

    def delete_provisioned_product(self, deployed):
        data = self.sc_client.terminate_provisioned_product(ProvisionedProductId=deployed.provisionedProductId)
        if not data:
            return False
        print "Provisioned product %s with version %s is being deprovisioned" % (deployed.provisionedProductName, deployed.productVersion)
        self.wait_for_terminate_provisioned_product(deployed,data)
        self.print_resources(data, deployed)
        return True

    ####### Cloudformation methods
    def get_stack_name(self,provisioning_product_data):
        data = self.describe_record(provisioning_product_data)
        while True:
            if "RecordOutputs" in data and len(data['RecordOutputs']) > 0:
                stackARN = data['RecordOutputs'][0]['OutputValue']
                m = re.search('/(SC(.*))/', stackARN)
                stackName=m.group(1)
                return stackName
            elif data["RecordDetail"]["Status"] == "FAILED":
                return None
            elif data["RecordDetail"]["Status"] == "SUCCEEDED":
                break
            else:
                data = self.describe_record(provisioning_product_data)

    def get_stackId(self,provisioning_product_data):
        data = self.describe_record(provisioning_product_data)
        while True:
            if "RecordOutputs" in data and len(data['RecordOutputs']) > 0:
                stackARN = data['RecordOutputs'][0]['OutputValue']
                return stackARN
            elif data["RecordDetail"]["Status"] == "FAILED":
                return None
            elif data["RecordDetail"]["Status"] == "SUCCEEDED":
                break
            else:
                data = self.describe_record(provisioning_product_data)

    def get_stack_events(self, deployed, provisioning_product_data, next_token=None):
        StackId = self.get_stackId(provisioning_product_data)
        if self._stack_exists(StackId):
            try:
                if next_token:
                    return self.cf_client.describe_stack_events(StackName=StackId, NextToken=next_token)
                else:
                    return self.cf_client.describe_stack_events(StackName=StackId)
            except ClientError, arg:
                print "WARN: Describe Stack Events generated exception.'%s'" % arg
                return None
        return None

    def print_events(self, deployed, provisioning_product_data):
        lastEventId = deployed.lastEventId
        new_events = []
        has_more = True
        next_token = None
        while has_more:
            all_events = self.get_stack_events(deployed, provisioning_product_data, next_token)
            if all_events == None:
                return True
            for event in all_events['StackEvents']:
                if lastEventId and event['EventId'] == lastEventId:
                    has_more = False
                    break
                else:
                    new_events.append(event)
            has_more = has_more and 'NextToken' in all_events and all_events['NextToken']
            if has_more:
                next_token = all_events['NextToken']
        if len(new_events):
            lastEventId = new_events[0]['EventId']
        for event in reversed(new_events):
            print self.get_string_event(event)
        deployed.lastEventId = lastEventId
        return True

    def get_string_event(self,event):
        event_data = "{}: {}".format(event['Timestamp'], event['ResourceStatus'])
        if 'LogicalResourceId' in event:
            event_data = "{} for {}".format(event_data, event['LogicalResourceId'])
        if 'ResourceStatusReason' in event:
            event_data = "{}. Reason: {}".format(event_data, event['ResourceStatusReason'])
        return event_data

    def _stack_exists(self, StackId):
        stacks = self.cf_client.list_stacks()['StackSummaries']
        for stack in stacks:
            if 'StackId' in stack and stack['StackId'] == StackId:
                return True

        return False

    def describe_stack(self, StackId):
        if self._stack_exists(StackId):
            try:
                return self.cf_client.describe_stacks(StackName=StackId)['Stacks'][0]
            except ClientError, arg:
                print "WARN: Describe Stack generated exception. The stack probably doesn't exist. '%s'" % arg
                return None
        return None

    def print_resources(self, provisioning_product_data, deployed):
        print "\nResources:"
        resources = self.list_resources(deployed, provisioning_product_data)
        if resources:
            for resource in resources:
                print "\n------------------------{}".format(self.get_string_resource(resource))
            print "\n------------------------"

    def get_string_resource(self, resource):
        resource_data = "\nType: {}".format(resource['ResourceType'])
        if 'LogicalResourceId' in resource:
            resource_data = "{}\nLogical Resource Id: {}".format(resource_data, resource['LogicalResourceId'])
        if 'PhysicalResourceId' in resource:
            resource_data = "{}\nPhysical Resource Id: {}".format(resource_data, resource['PhysicalResourceId'])
        if 'ResourceStatus' in resource:
            resource_data = "{}\nStatus: {}".format(resource_data, resource['ResourceStatus'])
        if 'LastUpdatedTimestamp' in resource:
            resource_data = "{}\nLast Updated: {}".format(resource_data, resource['LastUpdatedTimestamp'])
        return resource_data

    def capture_output(self, deployed):
        if self._stack_exists(deployed.stackId):
            stacks = self.cf_client.describe_stacks(StackName=deployed.stackId)['Stacks']
            if len(stacks) == 0:
                return False
            stack = stacks[0]
            output_variables = {}
            if "Outputs" in stack:
                outputs = stack['Outputs']
                for output in outputs:
                    output_variables[output['OutputKey']] = output['OutputValue']
            deployed.outputVariables = output_variables
            return True

        return False

    ####### Generic methods
    def _sanatize_name(self, name):
        p = re.compile('_| |\$|&|%|@|!|#')
        return p.sub('-', name)

    def prepare_provisioning_parameters(self, deployed):
        parameters = []
        for key, value in deployed.provisioningParameters.items():
            local_dict = {'Key': key, 'Value': value}
            parameters.append(local_dict)
        return parameters
